#include <stdio.h>
#include <sys/uio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>

#ifndef MAX_IOVEC	/* where the heck is it? */
#define MAX_IOVEC 1024
#endif

#define M_WRITE			0
#define M_FWRITE		1
#define M_FWRITE_UNLOCKED	2
#define M_WRITEV		3

char *progname;

struct line {
	char *data;
	int length;
	struct line *next;
};

struct line *read_one(FILE *f)
{
	char buf[1000];
	struct line *ret = 0;

	if (fgets(buf, sizeof(buf), f)) {
		ret = malloc(sizeof(*ret));
		ret->data = strdup(buf);
		ret->length = strlen(ret->data);
	}
	return ret;
}

struct line *read_in(FILE *f)
{
	struct line *ret = 0;
	struct line *last = 0;
	struct line *l;

	while ((l = read_one(f))) {
		if (ret == 0) {
			ret = l;
			l->next = 0;
			last = ret;
		} else {
			last->next = l;
			last = l;
			last->next = 0;
		}
	}
	return ret;
}

void write_out(int fd, struct line *line)
{
	while (line) {
		write(fd, line->data, line->length);
		line = line->next;
	}
}

void fwrite_out(FILE *f, struct line *line)
{
	while (line) {
		fwrite(line->data, line->length, 1, f);
		line = line->next;
	}
}

void fwrite_unlocked_out(FILE *f, struct line *line)
{
	while (line) {
		fwrite_unlocked(line->data, line->length, 1, f);
		line = line->next;
	}
}

void writev_out(int fd, struct line *line)
{
	struct iovec iov[MAX_IOVEC];
	int seg = 0;

	while (line) {
		if (seg == MAX_IOVEC) {
			writev(fd, iov, seg);
			seg = 0;
		}
		iov[seg].iov_base = line->data;
		iov[seg].iov_len = line->length;
		seg++;
		iov[seg].iov_base = 0;
		iov[seg].iov_len = 0;
		seg++;
		line = line->next;
	}
	if (seg)
		writev(fd, iov, seg);
}

void usage(void)
{
	fprintf(stderr, "Usage: %s [-fouw] [-n reps] infile outfile\n",
				progname);
	fprintf(stderr, "      -f:         Use fwrite()\n"); 
	fprintf(stderr, "      -o:         Use O_SYNC\n"); 
	fprintf(stderr, "      -t:         Don't use O_TRUNC\n"); 
	fprintf(stderr, "      -u:         Use fwrite_unlocked()\n"); 
	fprintf(stderr, "      -w:         Use writev()\n"); 
	exit(1);
}

int main(int argc, char *argv[])
{
	int reps = 100;
	int rep;
	int fd = 0;
	int c;
	struct line *lines;
	FILE *f;
	FILE *outf = NULL;
	int mode = M_WRITE;
	char *infile, *outfile;
	int o_sync = 0;
	int o_trunc = O_TRUNC;

	progname = argv[0];
	while ((c = getopt(argc, argv, "ftouwn:")) != -1) {
		switch (c) {
		case 'f':
			mode = M_FWRITE;
			break;
		case 'o':
			o_sync = O_SYNC;
			break;
		case 't':
			o_trunc = 0;
			break;
		case 'u':
			mode = M_FWRITE_UNLOCKED;
			break;
		case 'w':
			mode = M_WRITEV;
			break;
		case 'n':
			reps = strtol(optarg, NULL, 10);
			break;
		}
	}

	if (optind == argc)
		usage();
	infile = argv[optind++];
	if (optind == argc)
		usage();
	outfile = argv[optind++];
	if (optind != argc)
		usage();

	f = fopen(infile, "r");
	if (f == 0) {
		perror("open");
		exit(1);
	}
	lines = read_in(f);
	fclose(f);

	if (mode == M_FWRITE || mode == M_FWRITE_UNLOCKED) {
		outf = fopen(outfile, "w");
		if (outf == NULL) {
			perror("creat");
			exit(1);
		}
	} else {
		fd = open(outfile, O_RDWR|O_CREAT|o_sync|o_trunc, 0666);
		if (fd < 0) {
			perror("creat");
			exit(1);
		}
	}
	for (rep = 0; rep < reps; rep++) {
		switch (mode) {
		case M_WRITE:
			write_out(fd, lines);
			break;
		case M_FWRITE:
			fwrite_out(outf, lines);
			break;
		case M_FWRITE_UNLOCKED:
			fwrite_unlocked_out(outf, lines);
			break;
		case M_WRITEV:
			writev_out(fd, lines);
			break;
		}
	}
	exit(0);
}
