/*
 * http://marc.theaimsgroup.com/?l=linux-kernel&m=100230633324999&w=2
 */
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/mman.h>
#include <string.h>
#include <errno.h>

#define BLOCK_SIZE 42768352
// Adjust NUM_BLOCKS as needed to ensure that mlock won't fail
#define NUM_BLOCKS 4
// Sometimes when mmap fails at the default address it will succeed here
#define ALT_ADDR 0x0d000000
// We lock with 1M chunks to avoid machine lockups.
#define MLOCK_CHUNK 1048576
#define PAGE_SIZE 4096
char *addrs[NUM_BLOCKS];
const char *filepat = "chunk%d";

void mlock_slot(int i) {
  timeval start, end;
  gettimeofday(&start, NULL);
  char *top = addrs[i] + BLOCK_SIZE;
  printf("mlocking slot %i, %p\n", i, addrs[i]);
  for (char* addr = addrs[i]; addr < top;
       addr += MLOCK_CHUNK) {
    int size = addr + MLOCK_CHUNK <= top ?
               MLOCK_CHUNK : top - addr;
    if (addr == addrs[i])
      printf("mlocking at %p of size %d\n", addr, BLOCK_SIZE);
    if (mlock(addr, size) == -1) {
      printf("mlock failed: %s\n", strerror(errno));
      abort();
    } else {
      //      printf("mlock succeeded\n");
    }
  }
  gettimeofday(&end, NULL);
  long elapsed_usec = (end.tv_sec - start.tv_sec) * 1000000 +
                      (end.tv_usec - start.tv_usec);
  printf("mlock took %lf seconds\n", elapsed_usec / 1000000.0);
}

int main() {

  // First we setup our regions
  for (int i = 0; i < NUM_BLOCKS; ++i) {
    printf("mmap'ing %d bytes...\n", BLOCK_SIZE);
    addrs[i] = (char*)mmap(0, BLOCK_SIZE, PROT_READ, MAP_PRIVATE|MAP_ANON,
                           -1, 0);
    if (addrs[i] == MAP_FAILED) {
      printf("mmap failed: %s\n", strerror(errno));
      printf("trying backup region %x\n", ALT_ADDR);
      addrs[i] = (char*)mmap((char *)ALT_ADDR, BLOCK_SIZE, PROT_READ,
                             MAP_PRIVATE|MAP_ANON, -1, 0);
      if (addrs[i] == MAP_FAILED)
        printf("backup mmap failed: %s\n", strerror(errno));
    }
    if (addrs[i] != MAP_FAILED)
      printf("mmap'ed at %p-%lx.\n", addrs[i], (long)addrs[i] + BLOCK_SIZE);
  }

  // Okay, let's load over the old blocks
  for (int j = 0; j < 4 * NUM_BLOCKS; ++j) {
    int i = j % NUM_BLOCKS;
    if (addrs[i] == MAP_FAILED)
      break;
    if (j >= NUM_BLOCKS) { // We didn't mlock initially, don't munlock
      if ( munlock(addrs[i], BLOCK_SIZE) < 0 ) 
        printf("Error munlock'ing %p\n", addrs[i]);
      else
        printf("munlock'ed %p\n", addrs[i]);
    }
    if ( munmap(addrs[i], BLOCK_SIZE) < 0 )
      printf("Error munmap'ing region %p\n", addrs[i]);
    else
      printf("munmap'ed %p\n", addrs[i]);
    printf("Re-Loading data at %p for slot %i\n", addrs[i], i);
    char filename[1024];
    sprintf(filename, filepat, j);
    int fd = open(filename, O_RDONLY);
    if (fd < 0) abort();
    char *addr = (char*)mmap(addrs[i], BLOCK_SIZE, PROT_READ,
                             MAP_FIXED | MAP_PRIVATE, fd, 0);
    if (addr != addrs[i] || addr == MAP_FAILED) {
      printf("Load failed: %s\n", strerror(errno));
    } else {
      printf("Load (%s) succeeded!\n", filename);
    }
    mlock_slot(i);
    close(fd);
  }
  printf("sleeping...\n");
  while(1) sleep(5);
}
