#!/usr/bin/python
# *-*- Mode: Python -*-*

import gtk, gobject
import gtkvnc
import sys, os, commands, time, socket

# Replay

has_error = False
need_quit = True
override_need_quit = False

timeout_factor = 4
timeout_min = 60

button_mask = 0
pointer_x = 0
pointer_y = 0

def compare_frames(frame1, frame2):
    if ((frame1.get_width() != frame2.get_width()) or
        (frame1.get_height() != frame2.get_height())):
        return 0.0

    frame1_row = frame1.get_pixels()
    frame2_row = frame2.get_pixels()

    match = 0
    count = 0
    for y in range(frame1.get_height()):
        for x in range(frame1.get_width()):
            if ((frame1_row[x * 3 + 0] == frame2_row[x * 3 + 0]) and
                (frame1_row[x * 3 + 1] == frame2_row[x * 3 + 1]) and
                (frame1_row[x * 3 + 2] == frame2_row[x * 3 + 2])):
                match += 1
            count += 1
            
        frame1_row = frame1_row[frame1.get_rowstride():]
        frame2_row = frame2_row[frame2.get_rowstride():]

    return 100.0 * match / float(count)

def process_barriers(barriers, names, count, threshold, lines, vnc, time_delta, files):
    if count == 0:
        global has_error
        import random
        
        frame = vnc.get_pixbuf()
        num = random.randint(0, 999)
        failure = "failure-%03d.png" % num
        frame.save(failure, "png", {"tEXt::Generator App": "kvm-test"})
        print 'Failed to match barrier. Expected {%s}, got "%s"' % (', '.join(names), failure)
        has_error = True
        gtk.main_quit()
        
    frame = vnc.get_pixbuf()
    for barrier in barriers:
        match = compare_frames(barrier, frame)
        if match > threshold:
            print 'matched barrier'
            process_next_line(None, lines, vnc, time_delta + time.time(), files)
            return False
    
    print 'Could not match barrier, trying again', match
    gobject.timeout_add(2000, process_barriers,
                        barriers, names, count - 1, threshold, lines,
                        vnc, time_delta, files)
    return False

def process_next_line(parts, lines, vnc, time_delta, files):
    if parts:
        parts = map(lambda x: x.strip(), parts)
        if parts[0] == 'key':
           print parts[1], parts[2]
           if parts[2] == 'press':
              press = 1
           else:
              press = 0
              
           vnc.send_key_raw(gtk.gdk.keyval_from_name(parts[1]), press)
        elif parts[0] == 'barrier':
            global last_barrier_timestamp
            global timeout_factor
            global timeout_min
            
            timeout = (float(parts[3]) - last_barrier_timestamp) * timeout_factor
            timeout = max(long(timeout), timeout_min * timeout_factor)
            offset, match = parts[2].split(':')
            timeout += long(offset)
            print 'using %d second barrier timeout' % timeout
            barriers = []
            names = parts[1].split(':')
            for barrier in names:
                barriers.append(gtk.gdk.pixbuf_new_from_file(barrier))
            process_barriers(barriers, names, long(timeout) / 2, float(match), lines, vnc, time_delta - time.time(), files)
            return False
        elif parts[0] == 'change' and parts[1] == 'cdrom':
            qemu_change_cdrom(files[1], parts[2])
        elif parts[0] == 'eject' and parts[1] == 'cdrom':
            qemu_eject_cdrom(files[1])
        elif parts[0] == 'motion':
            global button_mask
            global pointer_x
            global pointer_y

            pointer_x = long(float(parts[1]))
            pointer_y = long(float(parts[2]))
            vnc.send_pointer(pointer_x, pointer_y, button_mask)
        elif parts[0] == 'button':
            global button_mask
            global pointer_x
            global pointer_y

            button = 1 << (long(parts[1]) - 1)
            if parts[2] == 'press':
                button_mask |= button
            elif parts[2] == 'release':
                button_mask &= ~button

            print 'button', button_mask
            vnc.send_pointer(pointer_x, pointer_y, button_mask)
        elif parts[0] == 'quit':
            gtk.main_quit()

    if len(lines) and len(lines[0]):
        parts = lines[0].split(',')
        timeout = float(parts[3]) - time_delta - time.time()
        timeout = long(timeout * 1000)

        # recompute time delta so that we're only looking at the difference
        # between steps, not necessarily the total difference since start
        time_delta = float(parts[3]) - time.time()

        if timeout <= 0:
            process_next_line(parts, lines[1:], vnc, time_delta, files)
        else:
            gobject.timeout_add(timeout, process_next_line,
                                parts, lines[1:], vnc, time_delta, files)
    
    return False

def vnc_initialized(src, window):
    if window:
        window.show_all()

def vnc_disconnected(src):
    global need_quit
    global override_need_quit

    if not override_need_quit:
        need_quit = False
    gtk.main_quit()

def launch_qemu(args):
    vnc_path = os.tmpnam()
    monitor = os.tmpnam()

    args = "%s -daemonize" % (args)
    args = "-monitor unix:%s,server,nowait %s" % (monitor, args)
    args = "-vnc unix:%s %s" % (vnc_path, args)

    s, o = commands.getstatusoutput('qemu-system-x86_64 %s' % args)
    if s != 0:
        sys.stdout.write(o)
        sys.stdout.write('\n')
        os.unlink(vnc_path)
        os.unlink(monitor)
        return s, []

    return 0, [vnc_path, monitor]

def quit_qemu(files):
    global need_quit
    
    if need_quit:
        fd = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)
        fd.connect(files[1])

        fd.send('quit\n')
        fd.close()

def replay_main(argv):
    global last_barrier_timestamp
    global timeout_min
    global timeout_factor
    from getopt import getopt

    opts, argv = getopt(argv,
                        "m:f:H",
                        ["timeout-min=", "timeout-factor="])
    for arg, value in opts:
        if arg in ['-m', '--timeout-min']:
            timeout_min = long(value)
        elif arg in ['-f', '--timeout-factor']:
            timeout_factor = long(value)

    log = open("vm.log", "r")
    lines = log.read().split('\n')

    s, files = launch_qemu(lines[0])
    if s:
        return s

    vnc = gtkvnc.Display()

    window = gtk.Window()
    window.set_resizable(False)
    window.set_title("kvm-test - Replay")
    window.add(vnc)
    window.connect('delete-event', window_delete)

    last_barrier_timestamp = float(lines[1])
    time_delta = float(lines[1]) - time.time()
    process_next_line(None, lines[2:], vnc, time_delta, files)

    vnc.realize()
    vnc.set_read_only(True)

    fd = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)
    fd.connect(files[0])
    vnc.fd = fd

    vnc.open_fd(fd.fileno())

    vnc.connect("vnc-initialized", vnc_initialized, window)
    vnc.connect("vnc-disconnected", vnc_disconnected)

    gtk.main()

    quit_qemu(files)

    for f in files:
        os.unlink(f)

    if has_error:
        return 1

    return 0

# Record

grabbed = False
barrier_count = 0
last_barrier_timestamp = 0

def set_title(vnc, window, grabbed):
    name = vnc.get_name()
    if grabbed:
        subtitle = "(Press Ctrl+Alt to release pointer) "
    else:
        subtitle = ""

    window.set_title("%s%s - kvm-test - Record" % (subtitle, name))

def do_add_barrier(src, vnc, log):
    global barrier_count

    pix = vnc.get_pixbuf()
    pix.save("screenshot-%03d.png" % barrier_count, "png",
             { "tEXt::Generator App": "kvm-test" })
    log.write("barrier, screenshot-%03d.png, %d:%f, %s\n" %
              (barrier_count, 0, 99.25, time.time()))
    barrier_count += 1

def qemu_change_cdrom(monitor_path, filename):
    s = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)
    s.connect(monitor_path)
    s.sendall('change cdrom "%s"\n' % filename)
    s.close()

def qemu_eject_cdrom(monitor_path):
    s = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)
    s.connect(monitor_path)
    s.sendall('eject cdrom\n')
    s.close()

chooser_cwd = None

def do_change_cdrom(src, monitor_path, log):
    global chooser_cwd
    
    chooser = gtk.FileChooserDialog(title="Choose CD-ROM",
                                    action=gtk.FILE_CHOOSER_ACTION_OPEN,
                                    buttons=(gtk.STOCK_CANCEL,
                                             gtk.RESPONSE_REJECT,
                                             gtk.STOCK_OK,
                                             gtk.RESPONSE_ACCEPT))
    if chooser_cwd:
        chooser.set_current_folder(chooser_cwd)
        
    rsp = chooser.run()
    if rsp == gtk.RESPONSE_ACCEPT:
        log.write('change, cdrom, %s, %s\n' %
                  (chooser.get_filename(),  time.time()))
        qemu_change_cdrom(monitor_path, chooser.get_filename())
        chooser_cwd = chooser.get_current_folder()
        
    chooser.destroy()

def do_eject_cdrom(src, monitor_path, log):
    log.write('eject, cdrom, 0, %s\n' % time.time())
    qemu_eject_cdrom(monitor_path)

def motion_event(src, ev, log, vnc):
    global grabbed
    if vnc.is_pointer_absolute() or grabbed:
        log.write("motion, %s, %s, %s\n" % (ev.x, ev.y, time.time()))
    return False

def button_event(src, ev, log, vnc):
    global grabbed

    if ev.type == gtk.gdk.BUTTON_PRESS:
        event = "press"
    else:
        event = "release"

    if vnc.is_pointer_absolute() or grabbed:
        log.write("button, %s, %s, %s\n" % (ev.button, event, time.time()))
    return False

def key_event(src, ev, log):
    if ev.type == gtk.gdk.KEY_PRESS:
        event = "press"
    else:
        event = "release"

    log.write("key, %s, %s, %s\n" % (gtk.gdk.keyval_name(ev.keyval), event,
                                     time.time()))
    return False

def vnc_grab(src, window):
    global grabbed
    grabbed = True
    set_title(src, window, True)

def vnc_ungrab(src, window):
    global grabbed
    grabbed = False
    set_title(src, window, False)

def record_vnc_initialized(src, window):
    set_title(src, window, False)
    window.show_all()

def window_delete(src, ev):
    global override_need_quit
    override_need_quit = True

def record_main(argv):
    args = '" "'.join(argv)
    if len(args):
        args = '"%s"' % args

    s, files = launch_qemu(args)
    if s:
        return s

    log = open("vm.log", "w", 0644)
    log.write("%s\n" % args)
    log.write("%s\n" % time.time())

    window = gtk.Window()
    vnc = gtkvnc.Display()
    window.set_resizable(False)

    layout = gtk.VBox()
    window.add(layout)

    menubar = gtk.MenuBar()
    barriers = gtk.MenuItem("_Barriers")
    menubar.append(barriers)
    devices = gtk.MenuItem("_Devices")
    menubar.append(devices)

    add_barrier = gtk.MenuItem("_Insert Barrier")
    change_cdrom = gtk.MenuItem("_Change CDROM")
    eject_cdrom = gtk.MenuItem("_Eject CDROM")

    submenu = gtk.Menu()
    submenu.append(add_barrier)
    barriers.set_submenu(submenu)

    submenu = gtk.Menu()
    submenu.append(change_cdrom)
    submenu.append(eject_cdrom)
    devices.set_submenu(submenu)
    
    add_barrier.connect("activate", do_add_barrier, vnc, log)
    change_cdrom.connect("activate", do_change_cdrom, files[1], log)
    eject_cdrom.connect("activate", do_eject_cdrom, files[1], log)

    layout.add(menubar)
    layout.add(vnc)

    vnc.realize()

    fd = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)
    fd.connect(files[0])
    vnc.fd = fd

    vnc.open_fd(fd.fileno())
    vnc.connect("vnc-pointer-grab", vnc_grab, window)
    vnc.connect("vnc-pointer-ungrab", vnc_ungrab, window)

    vnc.connect("vnc-initialized", record_vnc_initialized, window)
    vnc.connect("vnc-disconnected", vnc_disconnected)

    vnc.connect("key-press-event", key_event, log)
    vnc.connect("key-release-event", key_event, log)

    vnc.connect("button-press-event", button_event, log, vnc)
    vnc.connect("button-release-event", button_event, log, vnc)
    vnc.connect("motion-notify-event", motion_event, log, vnc)

    window.connect('delete-event', window_delete)

    gtk.main()

    quit_qemu(files)

    log.write("quit, 0, 0, %s\n" % time.time())
    log.close()

    for f in files:
        os.unlink(f)

    return 0
