/*
 * test_main.c: perfmon test suite core loop and functions
 *
 * Copyright (c) 2008 Google, Inc
 * Contributed by Stephane Eranian <eranian@google.com>
 */
#ifndef _GNU_SOURCE
#define _GNU_SOURCE /* for getline */
#endif
#include <sys/types.h>
#include <inttypes.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>
#include <sys/utsname.h>

#include <perfmon/pfmlib.h>
#include "pfm_tests.h"

/*
 * external declaration for test
 */
DECL_TEST(sysfs);
DECL_TEST(self);
DECL_TEST(notify_self);
DECL_TEST(notify_self_siginfo);
DECL_TEST(pfm_create_context);
DECL_TEST(pfm_write_pmcs);
DECL_TEST(pfm_write_pmds);
DECL_TEST(pfm_read_pmds);
DECL_TEST(pfm_load_context);
DECL_TEST(system_wide);
DECL_TEST(task);
DECL_TEST(mmap);
DECL_TEST(task_smpl);
DECL_TEST(self_smpl);
DECL_TEST(multiplex1);
DECL_TEST(multiplex2);

/*
 * table of tests
 */
static test_func_t test_funcs[]={
	PFM_TEST(sysfs),
	PFM_TEST(pfm_create_context),
	PFM_TEST(pfm_write_pmcs),
	PFM_TEST(pfm_write_pmds),
	PFM_TEST(pfm_read_pmds),
	PFM_TEST(pfm_load_context),
	PFM_TEST(system_wide),
	PFM_TEST(task),
	PFM_TEST(mmap),
	PFM_TEST(self),
	PFM_TEST(notify_self),
	PFM_TEST(notify_self_siginfo),
	PFM_TEST(task_smpl),
	PFM_TEST(self_smpl),
	PFM_TEST(multiplex1),
	PFM_TEST(multiplex2),
	{ "", NULL}
};

int pfmlib_ok;	/* non-zero if libpfm initialized correctly */

/*
 * write val into file fn
 */
int
write_sysfs(char *fn, char *val)
{
	FILE *fp;
	int ret;

	fp = fopen(fn, "w");
	if (!fp) {
		PFM_LOG("cannot open %s", fn);
		return -1;
	}
	ret = fprintf(fp, val);
	if (ret != strlen(val)) {
		PFM_LOG("cannot write %s in %s", val, fn);
	}
	fclose(fp);
	return ret > -1 ? 0 : -1;
}

/*
 * return first line of file fn. supposed to be used with sysfs only
 * string is dynamically allocated and MUST be freed when not needed
 * nymore
 */
char *
read_sysfs(char *fn)
{
	FILE *fp;
	char *line = NULL;
	size_t len = 0;
	ssize_t ret;

	fp = fopen(fn, "r");
	if (!fp) {
		PFM_LOG("cannot open %s", fn);
		return NULL;
	}
	ret = getline(&line, &len, fp);
	if (ret == -1)
		PFM_LOG("cannot read date from %s: %s", fn, strerror(errno));

	fclose(fp);

	return line;
}

static int
check_version(void)
{
	struct utsname uts;
	char *p, *endptr = NULL;
	unsigned long m;
	int ret;

	ret = uname(&uts);
	if (ret == -1) {
		PFM_LOG("uname failed: %s", strerror(errno));
		return -1;
	}
	PFM_LOG("version=%s", uts.release);
	p = strtok(uts.release, ".");
	if (!p) {
		PFM_LOG("parsing release failed: %s", uts.release);
		return -1;
	}
	if (strcmp(p, "2")) {
		PFM_LOG("invalid release: %s", uts.release);
		return -1;
	}
	p = strtok(NULL, ".");
	if (!p) {
		PFM_LOG("parsing release failed: %s", uts.release);
		return -1;
	}	
	if (strcmp(p, "6")) {
		PFM_LOG("invalid release: %s", uts.release);
		return -1;
	}

	p = strtok(NULL, ".-");
	if (!p) {
		PFM_LOG("parsing release failed: %s", uts.release);
		return -1;
	}	
	m = strtoul(p, &endptr, 10);
	if (*endptr != '\0') {
		PFM_LOG("parsing release failed: %s", uts.release);
		return -1;
	}
	if (m < 26)
		PFM_LOG("test skipped because needs at least 2.6.26");
	/* SKIP (-2) if invalid version */
	return m < 26 ? -2 : 0;
}


int
main(int argc, char **argv)
{
	char **argv_orig;
	test_func_t *p;
	unsigned int num_errors;
	int ret;

	ret = check_version();
	if (ret) {
		printf("test_main         [%s]\n", ret == -1 ? "FAIL" : ret == -2 ? "SKIP" : "PASS");
		return -1;
	}
	/*
 	 * initialize libpfm
 	 * it is okay to fail here, it's just that some
 	 * tests will not work
 	 */
	ret = pfm_initialize();
	pfmlib_ok = ret == PFMLIB_SUCCESS;

	num_errors = 0;

	/*
	 * keep track or original argv vector
	 * needed to get to argv[0] in certain
	 * tests
	 */
	argv_orig = argv;

	/*
	 * check if user passed a list of tests
	 */
	if (argc > 1) {
		while (*++argv) {
			for (p = test_funcs; p->func; p++) {
				if (strcmp(p->name, *argv))
					continue;

				ret = (*p->func)(argc, argv_orig);

				if (ret)
					num_errors++;
			}
		}
	} else {
		/* execute all tests */
		for (p = test_funcs; p->func; p++) {
			ret = (*p->func)(argc, argv_orig);

			if (ret)
				num_errors++;
		}
	}
	return num_errors ? -1 : 0;
}
