/*
 * test_write_pmcd.c - test pfm_write_pmds
 *
 * Copyright (c) 2008 Google, Inc
 * Contributed by Stephane Eranian <eranian@google.com>
 */
#include <sys/types.h>
#include <inttypes.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <sys/wait.h>
#include <sys/mman.h>

#include <perfmon/perfmon.h>
#include <perfmon/perfmon_dfl_smpl.h>
#include <perfmon/pfmlib.h>

#include "pfm_tests.h"
#include "detect_pmcs.h"

static int
get_avail_reg(pfmlib_regmask_t *unavail)
{
	int i;
	/* look for first available register */
	for (i=0; i < PFMLIB_REG_MAX; i++) {
		if (!pfm_regmask_isset(unavail, i))
			break;
	}
	return i == PFMLIB_REG_MAX ? -1 : i;
}

static int
test1(int fd, pfmlib_regmask_t *unavail)
{
	int ret;

	/* null vector, null size */
	ret = pfm_write_pmds(fd, NULL, 0);
	if (ret == -1) {
		PFM_LOG("unexpected error: %s", strerror(errno));
		return -1;
	}
	return 0;
}

static int
test2(int fd, pfmlib_regmask_t *unavail)
{
	int ret;

	/* null vector */
	ret = pfm_write_pmds(fd, NULL, sizeof(pfarg_pmd_t));
	if (ret == 0) {
		PFM_LOG("should not have accepted write pmds");
		return -1;
	}
	return 0;
}


static int
test3(int fd, pfmlib_regmask_t *unavail)
{
	int ret;
	pfarg_pmd_t pd[1];

	memset(pd, 0, sizeof(pd));

	/* null size */
	ret = pfm_write_pmds(fd, pd, 0);
	if (ret == -1) {
		PFM_LOG("unexpected error: %s", strerror(errno));
		return -1;
	}
	return 0;
}

static int
test5(int fd, pfmlib_regmask_t *unavail)
{
	int ret;
	pfarg_pmd_t pd[1];

	memset(pd, 0, sizeof(pd));

	/* use a non perfmon-related fd */
	ret = pfm_write_pmds(0, pd, 1);
	if (ret == 0) {
		PFM_LOG("should not have accepted pmcs write");
		return -1;
	}
	if (errno != EBADF) {
		PFM_LOG("unexpected errno=%d vs EBADF", errno);
		return -1;
	}
	return 0;
}

static int
test6(int fd, pfmlib_regmask_t *unavail)
{
	int i, ret, avail;
	pfarg_pmd_t *pd;
	unsigned long max;
	size_t n;
	char *str;

	str = read_sysfs("/sys/kernel/perfmon/arg_mem_max");
	if (!str) {
		PFM_LOG("cannot read arg_mem_max");
		return -1;
	}

	avail = get_avail_reg(unavail);
	if (avail == -1) {
		PFM_LOG("no available registers");
		return -1;
	}

	max = strtoul(str, NULL, 0);
	free(str);
	n = max / sizeof(pfarg_pmd_t);
	n++;

	pd = calloc(n, sizeof(pfarg_pmd_t));
	if (!pd) {
		PFM_LOG("failed to allocate space for %zu pfarg_pmd_t",n);
		free(str);
		return -1;
	}

	memset(pd, 0, n * sizeof(pfarg_pmd_t));
	/* make sure we use a valid register */
	for (i=0; i < n ; i++)
		pd[i].reg_num = avail;

	/* vector is too large */
	ret = pfm_write_pmds(fd, pd, n);
	free(pd);
	if (ret == -1) {
		if (errno != E2BIG) {
			PFM_LOG("unexpected errno: %s", strerror(errno));
			return -1;
		}
		return 0;
	}
	PFM_LOG("should not have accepted pmds write");
	return -1;
}

static int
test7(int fd, pfmlib_regmask_t *unavail)
{
	pfarg_pmd_t pd[1];
	int i, ret;

	/* look for unimplemented/unavailable register */
	for (i=0; i < PFMLIB_REG_MAX; i++) {
		if (pfm_regmask_isset(unavail, i))
			goto found;
	}
	PFM_LOG("cannot find unimplemented registers, skipping test");
	return 0;
found:
	memset(pd, 0, sizeof(pd));
	pd[0].reg_num = i;

	/* try writing unavailable pmc */
	ret = pfm_write_pmds(fd, pd, 1);
	if (ret == -1) {
		if (errno != EINVAL) {
			PFM_LOG("unexpected errno: %s", strerror(errno));
			return -1;
		}
		return 0;
	}
	PFM_LOG("should not have accepted pmd write");
	return -1;
}

static int
test8(int fd, pfmlib_regmask_t *unavail)
{
	pfarg_pmd_t *pd;
	size_t pgsz;
	void *addr;
	int ret, avail;

	avail = get_avail_reg(unavail);
	if (avail == -1) {
		PFM_LOG("no available registers");
		return -1;
	}

	pgsz = getpagesize();

	addr = mmap(NULL, 2 * pgsz, PROT_READ|PROT_WRITE, MAP_PRIVATE|MAP_ANONYMOUS, -1, 0);
	if (addr == MAP_FAILED) {
		PFM_LOG("mmap failed: %s", strerror(errno));
		return -1;
	}
	ret = munmap(addr + pgsz, pgsz);
	if (ret == -1) {
		PFM_LOG("munmap failed: %s", strerror(errno));
		return -1;
	}
	pd = addr +pgsz - sizeof(*pd);

	memset(pd, 0, sizeof(pd));

	pd[0].reg_num = avail;

	/* pass invalid memory (2nd element of vector) */
	ret = pfm_write_pmds(fd, pd, 2);
	if (ret == 0) {
		PFM_LOG("should not have accepted write pmds");
		munmap(addr, pgsz);
		return -1;
	}
	if (ret == -1 && errno != EFAULT) {
		PFM_LOG("unexpected errno=%d vs. EFAULT", errno);
	} else
		ret = 0;
	munmap(addr, pgsz);
	return ret;
}

static int
test9(int fd, pfmlib_regmask_t *unavail)
{
	pfmlib_input_param_t inp;
	pfmlib_output_param_t outp;
	pfarg_pmc_t pc[8];
	pfarg_pmd_t pd[8];
	int i, ret;
	
	/* ignore if libpfm not initialized */
	if (pfmlib_ok == 0)
		return -2;

	memset(&inp,0, sizeof(inp));
	memset(&outp,0, sizeof(outp));
	memset(pc, 0, sizeof(pc));
	memset(pd, 0, sizeof(pd));

	ret = pfm_get_cycle_event(&inp.pfp_events[0]);
	if (ret != PFMLIB_SUCCESS) {
		PFM_LOG("cannot find cycle event: %s", pfm_strerror(ret));
		return -1;
	}
	inp.pfp_dfl_plm = PFM_PLM3;
	inp.pfp_event_count = 1;
	inp.pfp_unavail_pmcs = *unavail;

	ret = pfm_dispatch_events(&inp, NULL, &outp, NULL);
	if (ret != PFMLIB_SUCCESS) {
		PFM_LOG("failed dispatch_events: %s", pfm_strerror(ret));
		return -1;
	}

	if (outp.pfp_pmc_count == 0) {
		PFM_LOG("errno no pmc to write");
		return -1;
	}

	if (outp.pfp_pmc_count*sizeof(pfarg_pmc_t) > sizeof(pc)) {
		PFM_LOG("pc vector is too small");
		return -1;
	}

	for (i=0; i < outp.pfp_pmc_count; i++) {
		pc[i].reg_num   = outp.pfp_pmcs[i].reg_num;
		pc[i].reg_value = outp.pfp_pmcs[i].reg_value;
	}

	for (i=0; i < outp.pfp_pmd_count; i++) {
		pd[i].reg_num   = outp.pfp_pmds[i].reg_num;
	}

	if (pfm_write_pmds(fd, pd, outp.pfp_pmd_count)) {
		PFM_LOG("cannot write pmd: %s", strerror(errno));
		return -1;
	}

	/* check RANDOM flag */
	pd[0].reg_flags = PFM_REGFL_RANDOM;
	if (pfm_write_pmds(fd, pd, 1)) {
		PFM_LOG("cannot write pmd with RANDOM flag: %s", strerror(errno));
		return -1;
	}

	/* check OVFL_NOTIFY flag */
	pd[0].reg_flags = PFM_REGFL_OVFL_NOTIFY;
	if (pfm_write_pmds(fd, pd, 1)) {
		PFM_LOG("cannot write pmd with OVFL_NOTIFY flag: %s", strerror(errno));
		return -1;
	}
	/* check RANDOM+OVFL_NOTIFY flag */
	pd[0].reg_flags = PFM_REGFL_OVFL_NOTIFY| PFM_REGFL_RANDOM;
	if (pfm_write_pmds(fd, pd, 1)) {
		PFM_LOG("cannot write pmd with OVFL_NOTIFY flag: %s", strerror(errno));
		return -1;
	}
	return 0;
}

static int
test10(int fd, pfmlib_regmask_t *unavail)
{
	pfarg_pmd_t pd[1];
	int ret, avail, i;

	memset(pd, 0, sizeof(pd));

	avail = get_avail_reg(unavail);
	if (avail == -1) {
		PFM_LOG("no available registers");
		return -1;
	}

	pd[0].reg_num = avail;

	for (i=0; i < PFMLIB_REG_MAX; i++) {
		if (pfm_regmask_isset(unavail,i)) {
			pfm_bv_set(pd[0].reg_smpl_pmds, i);
			goto found;
		}
	}
	PFM_LOG("not unavailable PMD registers, skipping test");
	return -2;
found:
	/* pass invalid smpl_pmds */
	ret = pfm_write_pmds(fd, pd, 1);
	if (ret == 0) {
		PFM_LOG("should not have accepted write pmds");
		return -1;
	}
	if (ret == -1 && errno != EINVAL) {
		PFM_LOG("unexpected errno=%d vs. EINVAL", errno);
	}
	memcpy(pd[0].reg_reset_pmds, pd[0].reg_smpl_pmds, sizeof(pd[0].reg_smpl_pmds));

	/* pass invalid reset_pmds */
	ret = pfm_write_pmds(fd, pd, 1);
	if (ret == 0) {
		PFM_LOG("should not have accepted write pmds");
		return -1;
	}
	if (ret == -1 && errno != EINVAL) {
		PFM_LOG("unexpected errno=%d vs. EINVAL", errno);
	} else
		ret = 0;

	return ret;
}

static int
test11(int fd, pfmlib_regmask_t *unavail)
{
	pfarg_pmd_t pd[1];
	pfmlib_regmask_t impl_cnts;
	int ret, avail, i, min;

	/* ignore if libpfm is not initialized */
	if (pfmlib_ok == 0)
		return -2;

	memset(pd, 0, sizeof(pd));

	avail = get_avail_reg(unavail);
	if (avail == -1) {
		PFM_LOG("no available registers");
		return -1;
	}

	pd[0].reg_num = avail;

	min = PFMLIB_REG_MAX < PFM_MAX_PMDS ? PFMLIB_REG_MAX : PFM_MAX_PMDS;

	for (i=0; i < min; i++) {
		if (!pfm_regmask_isset(unavail,i)) {
			pfm_bv_set(pd[0].reg_smpl_pmds, i);
		}
	}	

	/* pass valid smpl_pmds : read-only registers are fine here */
	ret = pfm_write_pmds(fd, pd, 1);
	if (ret == -1) {
		PFM_LOG("unexpected error: %s", strerror(errno));
		return -1;
	}

	if (pfmlib_ok == 0)
		return 0;

	memcpy(pd[0].reg_reset_pmds, pd[0].reg_smpl_pmds, sizeof(pd[0].reg_smpl_pmds));

	pfm_get_impl_counters(&impl_cnts);

	/* only keep pmds we know are read-write (counters) */
	for(i=0; i < min; i++) {
		if (!pfm_regmask_isset(&impl_cnts, i)) {
			pfm_bv_clear(pd[0].reg_reset_pmds, i);
		}
	}

	/* pass valid reset_pmds: only read-write registers are fine here */
	ret = pfm_write_pmds(fd, pd, 1);
	if (ret == -1) {
		PFM_LOG("unexpected error: %s", strerror(errno));
	}
	return ret;
}

/*
 * tests of pfm_write_pmcs() after pfm_load_context() are deferred to
 * a series after the tests for pfm_load_context
 */

static int (*ctx_tests[])(int fd, pfmlib_regmask_t *unavail)={
	test1,
	test2,
	test3,
	/* test4 is empty */
	test5,
	test6,
	test7,
	test8,
	test9,
	test10,
	test11,
	NULL
};

int
main_pfm_write_pmds(int argc, char **argv)
{
	pfarg_ctx_t ctx;
	pfmlib_regmask_t unavail_pmcs, unavail_pmds;
	int ctx_fd, ret = -1, i;
	int (**pf)(int fd, pfmlib_regmask_t *unavail);


	memset(&ctx, 0, sizeof(ctx));

	ctx_fd = pfm_create_context(&ctx, NULL, NULL, 0);
	if (ctx_fd == -1)  {
		PFM_LOG("unexpected context creation error: %s", strerror(errno));
		return -1;
	}

	memset(&unavail_pmcs, 0, sizeof(unavail_pmcs));
	memset(&unavail_pmds, 0, sizeof(unavail_pmds));

	/* ignroe return value: best effort */
	detect_unavail_pmu_regs(ctx_fd, &unavail_pmcs, &unavail_pmds);

	for (pf = ctx_tests, i=0; *pf ; pf++, i++) {
		printf("pfm_write_pmds.test%-3d ", i); fflush(stdout);
		ret = (*pf)(ctx_fd, &unavail_pmds);
		printf("[%s]\n", ret == -1 ? "FAIL" : ret == -2 ? "SKIP" : "PASS");
		if (ret)
			break;
	}

	close(ctx_fd);

	return ret;
}
