/*
 * test_task_smpl.c - test sampling of another process
 *
 * Copyright (c) 2008 Google, Inc
 * Contributed by Stephane Eranian <eranian@google.com>
 * 
 * Based on:
 *
 * Copyright (c) 2003-2006 Hewlett-Packard Development Company, L.P.
 * Contributed by Stephane Eranian <eranian@hpl.hp.com>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>
#include <stdarg.h>
#include <stdint.h>
#include <time.h>
#include <sys/ptrace.h>
#include <sys/wait.h>
#include <sys/mman.h>
#include <sys/time.h>
#include <sys/resource.h>

#include <perfmon/pfmlib.h>
#include <perfmon/perfmon.h>
#include <perfmon/perfmon_dfl_smpl.h>

#include "detect_pmcs.h"
#include "pfm_tests.h"

#define SMPL_PERIOD	100000

typedef pfm_dfl_smpl_arg_t		smpl_fmt_arg_t;
typedef pfm_dfl_smpl_hdr_t		smpl_hdr_t;
typedef pfm_dfl_smpl_entry_t		smpl_entry_t;
typedef pfm_dfl_smpl_arg_t		smpl_arg_t;
#define FMT_NAME			PFM_DFL_SMPL_NAME

#define NUM_PMCS PFMLIB_MAX_PMCS
#define NUM_PMDS PFMLIB_MAX_PMDS


#define TEST_PRG "noploop"

static uint64_t num_entries;

static int
child(char *cmdline)
{
	size_t len;
	char *buf, *p;

	len = strlen(cmdline);
	len += 1 + strlen(TEST_PRG) + 1;

	buf = malloc(len);
	if (!buf) {
		PFM_LOG("cannot allocate memory for %zu bytes for string", len);
		exit(1);
	}

	buf[0] = '\0';
	p = strrchr(cmdline, '/');
	if (p) {
		strncpy(buf, cmdline, ++p - cmdline);
	} 
	/* make sure we grab TEST_PRG from the same dir used for this process */
	strcat(buf, TEST_PRG);

	/*
	 * force the task to stop before executing the first
	 * user level instruction
	 */
	ptrace(PTRACE_TRACEME, 0, NULL, NULL);
	close(0); close(1); close(2);
	execl(buf, buf, "10", NULL);
	/* not reached */
	exit(1);
}

static int
process_smpl_buf(pid_t pid, pid_t tid, smpl_hdr_t *hdr, uint16_t ovfl_pmd)
{
	static uint64_t last_overflow = ~0; /* initialize to biggest value possible */
	static uint64_t last_count;
	smpl_entry_t *ent;
	size_t pos, count, entry_size;
	uint64_t *reg;
	uint64_t prev_tstamp, old_reg;
	
	/* check  for already processed buffer */
	if (hdr->hdr_overflows == last_overflow && hdr->hdr_count == last_count)
		return 0;

	count = hdr->hdr_count;

	ent   = (smpl_entry_t *)(hdr+1);
	pos   = (unsigned long)ent;

	entry_size = sizeof(*ent) + 1 * sizeof(uint64_t);

	prev_tstamp = old_reg = 0;


	while(count--) {
		if (ent->tgid != pid) {
			PFM_LOG("unexpected pid=%d vs. %d", ent->tgid, pid);
			return -1;
		}

		if (ent->pid != tid) {
			PFM_LOG("unexpected tid=%d vs. %d", ent->pid, tid);
			return -1;
		}

		if (ent->last_reset_val != -SMPL_PERIOD) {
			PFM_LOG("unexpected last_val=%"PRIu64" vs. %lu", ent->last_reset_val, (unsigned long)-SMPL_PERIOD);
			return -1;
		}

		if (ent->ovfl_pmd != ovfl_pmd) {
			PFM_LOG("unexpected ovfl pmd=%u vs %u", ent->ovfl_pmd, ovfl_pmd);
			return -1;
		}

		if (ent->set != 0) {
			PFM_LOG("invalid set=%u vs. 0", ent->set);
			return -1;
		}
		if (ent->tstamp <= prev_tstamp) {
			PFM_LOG("unexpected tstamp=%"PRIu64" should be >  %"PRIu64, ent->tstamp, prev_tstamp);
			return -1;
		}
		prev_tstamp = ent->tstamp;
		/* should also test tstamp: monotonically increasing */

		/*
		 * print body: additional PMDs recorded
		 * PMD are recorded in increasing index order
		 */
		reg = (uint64_t *)(ent+1);

		if (*reg < old_reg) {
			PFM_LOG("unexpected smpl pmd value=%"PRIu64" should be > %"PRIu64, *reg, old_reg);
			return -1;
		}
		old_reg = *reg;

		pos += entry_size;
		ent = (smpl_entry_t *)pos;
		num_entries++;
	}
	return 0;
}

static int
do_task_smpl(char *cmdline)
{
	smpl_hdr_t *hdr = NULL;
	pfarg_ctx_t ctx;
	smpl_arg_t buf_arg;
	pfmlib_input_param_t inp;
	pfmlib_output_param_t outp;
	pfarg_pmd_t pd[NUM_PMDS];
	pfarg_pmc_t pc[NUM_PMCS];
	pfarg_load_t load_args;
	pfarg_msg_t msg;
	void *buf_addr;
	pid_t pid;
	int status, ret, fd;
	unsigned int i, num_counters;

	/*
	 * intialize all locals
	 */
	memset(&ctx, 0, sizeof(ctx));
	memset(&buf_arg, 0, sizeof(buf_arg));
	memset(&inp,0, sizeof(inp));
	memset(&outp,0, sizeof(outp));
	memset(pd, 0, sizeof(pd));
	memset(pc, 0, sizeof(pc));
	memset(&load_args, 0, sizeof(load_args));

	ret = pfm_get_num_counters(&num_counters);
	if (ret != PFMLIB_SUCCESS) {
		PFM_LOG("cannot get number of coutners: %s", pfm_strerror(ret));
		return -1;
	}

	ret = pfm_get_cycle_event(&inp.pfp_events[0]);
	if (ret != PFMLIB_SUCCESS) {
		PFM_LOG("cannot find cycle event: %s", pfm_strerror(ret));
		return -1;
	}

	ret = pfm_get_inst_retired_event(&inp.pfp_events[1]);
	if (ret != PFMLIB_SUCCESS) {
		PFM_LOG("cannot find inst retired event: %s", pfm_strerror(ret));
		return -1;
	}

	i = 2;

	/*
	 * set the privilege mode:
	 * 	PFM_PLM3 : user level
	 * 	PFM_PLM0 : kernel level
	 */
	inp.pfp_dfl_plm   = PFM_PLM3;

	if (i > num_counters) {
		i = num_counters;
		PFM_LOG("too many events provided (max=%d events), using first %d event(s)", num_counters, i);
		return -1;
	}
	inp.pfp_event_count = i;

	detect_unavail_pmcs(-1, &inp.pfp_unavail_pmcs);

	if ((ret=pfm_dispatch_events(&inp, NULL, &outp, NULL)) != PFMLIB_SUCCESS) {
		PFM_LOG("cannot configure events: %s", pfm_strerror(ret));
		return -1;
	}

	for (i=0; i < outp.pfp_pmc_count; i++) {
		pc[i].reg_num   = outp.pfp_pmcs[i].reg_num;
		pc[i].reg_value = outp.pfp_pmcs[i].reg_value;
	}

	for (i=0; i < outp.pfp_pmd_count; i++) {
		pd[i].reg_num = outp.pfp_pmds[i].reg_num;

		/*
		 * skip first counter (sampling period)
		 */
		if (i) {
			pfm_bv_set(pd[0].reg_smpl_pmds, pd[i].reg_num);
		}
	}

	pd[0].reg_flags	|= PFM_REGFL_OVFL_NOTIFY;

	/* set sampling periods */
	pd[0].reg_value       = - SMPL_PERIOD;
	pd[0].reg_short_reset = - SMPL_PERIOD;
	pd[0].reg_long_reset  = - SMPL_PERIOD;

	buf_arg.buf_size = 2*getpagesize();

	fd = pfm_create_context(&ctx, FMT_NAME, &buf_arg, sizeof(buf_arg));
	if (fd == -1) {
		PFM_LOG("cannot create context: %s", strerror(errno));
		return -1;
	}

	/*
	 * retrieve the virtual address at which the sampling
	 * buffer has been mapped
	 */
	buf_addr = mmap(NULL, (size_t)buf_arg.buf_size, PROT_READ, MAP_PRIVATE, fd, 0);
	if (buf_addr == MAP_FAILED) {
		PFM_LOG("cannot mmap sampling buffer: %s", strerror(errno));
		goto error;
	}

	hdr = (smpl_hdr_t *)buf_addr;

	if (pfm_write_pmcs(fd, pc, outp.pfp_pmc_count)) {
		PFM_LOG("pfm_write_pmcs error errno: %s", strerror(errno));
		goto error;
	}
	/*
	 * initialize the PMDs
	 * To be read, each PMD must be either written or declared
	 * as being part of a sample (reg_smpl_pmds, reg_reset_pmds)
	 */
	if (pfm_write_pmds(fd, pd, outp.pfp_pmd_count)) {
		PFM_LOG("pfm_write_pmds error errno: %s", strerror(errno));
		goto error;
	}

	if ((pid=fork()) == -1) {
		PFM_LOG("cannot fork process: %s", strerror(errno));
		goto error;
	}

	if (pid == 0) {
		close(fd);
		child(cmdline);
	}

	waitpid(pid, &status, WUNTRACED);

	if (WIFEXITED(status)) {
		PFM_LOG("process [%d] exited already status %d", pid, WEXITSTATUS(status));
		goto terminate_session;
	}

	/*
	 * attach context to stopped task
	 */
	load_args.load_pid = pid;
	if (pfm_load_context (fd, &load_args)) {
		PFM_LOG("pfm_load_context error: %s", strerror(errno));
		ptrace(PTRACE_DETACH, pid, NULL, 0);
		goto error_kill;
	}

	if (pfm_start(fd, NULL)) {
		PFM_LOG("pfm_start error: %s", strerror(errno));
		ptrace(PTRACE_DETACH, pid, NULL, 0);
		goto error_kill;
	}

	/* effectively start monitoring */
	ptrace(PTRACE_DETACH, pid, NULL, 0);

	for(;;) {
		ret = read(fd, &msg, sizeof(msg));
		if (ret == -1) {
			if(ret == -1 && errno == EINTR)
				continue;

			PFM_LOG("cannot read perfmon msg: %s", strerror(errno));
		}

		switch(msg.type) {
			case PFM_MSG_OVFL: /* the sampling buffer is full */
				ret = process_smpl_buf(pid, pid, hdr, pd[0].reg_num);
				if (ret)
					goto error_kill;

				ret = pfm_restart(fd);
				if (ret && ret != EBUSY) {
					PFM_LOG("pfm_restart error: %s", strerror(errno));
					goto error_kill;
				}
				break;

			case PFM_MSG_END: /* monitored task terminated */
				goto terminate_session;

			default: PFM_LOG("unknown message type %d", msg.type);
				 goto error_kill;
		}
	}
terminate_session:
	/*
	 * cleanup child
	 */
	waitpid(pid, &status, 0);

	process_smpl_buf(pid, pid, hdr, pd[0].reg_num);

	close(fd);

	if (!num_entries) {
		PFM_LOG("did not capture any sample");
		goto error_kill;
	}

	ret = munmap(hdr, (size_t)buf_arg.buf_size);
	if (ret) {
		PFM_LOG("cannot unmap buffer: %s", strerror(errno));
		goto error_kill;
	}
	return 0;
error_kill:
	kill(pid, SIGKILL);
	waitpid(pid, &status, 0);
error:
	if (hdr)
		munmap(hdr, (size_t)buf_arg.buf_size);
	close(fd);
	return -1;
}

int
main_task_smpl(int argc, char **argv)
{
	int ret;

	/* need library for events */
	if (pfmlib_ok == 0)
		return -2;

	printf("test_task_smpl.test0  "); fflush(stdout);
	ret = do_task_smpl(*argv);
	printf("[%s]\n", ret == -1 ? "FAIL" : ret == -2 ? "SKIP" : "PASS");
	return ret;
}
