/*
 * test_self_smpl.c - test self sampling with kernel buffer
 *
 * Copyright (c) 2008 Google, Inc
 * Contributed by Stephane Eranian <eranian@google.com>
 * 
 * Based on:
 *
 * Copyright (c) 2003-2006 Hewlett-Packard Development Company, L.P.
 * Contributed by Stephane Eranian <eranian@hpl.hp.com>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <stdarg.h>
#include <stdint.h>
#include <setjmp.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <sys/time.h>
#include <sys/resource.h>

#include <perfmon/pfmlib.h>
#include <perfmon/perfmon.h>
#include <perfmon/perfmon_dfl_smpl.h>

#include "detect_pmcs.h"
#include "pfm_tests.h"

#define SMPL_PERIOD	100000

typedef pfm_dfl_smpl_arg_t		smpl_fmt_arg_t;
typedef pfm_dfl_smpl_hdr_t		smpl_hdr_t;
typedef pfm_dfl_smpl_entry_t		smpl_entry_t;
typedef pfm_dfl_smpl_arg_t		smpl_arg_t;
#define FMT_NAME			PFM_DFL_SMPL_NAME

#define NUM_PMCS PFMLIB_MAX_PMCS
#define NUM_PMDS PFMLIB_MAX_PMDS


#define NOPLOOP_DELAY_SECS	5 /* seconds */

static uint64_t num_entries;

static volatile int not_done = 1;
static jmp_buf jbuf;

static int ctx_fd, error_code;
static uint16_t ovfl_pmd;
static smpl_hdr_t *hdr;

void alarm_handler(int sig)
{
	not_done = 0;
}

static void
noploop()
{
	signal(SIGALRM, alarm_handler);
	alarm(NOPLOOP_DELAY_SECS);
	for(;not_done;);
}

/* called from signal handler context, careful with functions used */
static int
process_smpl_buf(pid_t pid, pid_t tid, smpl_hdr_t *hdr, uint16_t ovfl_pmd)
{
	static uint64_t last_overflow = ~0; /* initialize to biggest value possible */
	static uint64_t last_count;
	smpl_entry_t *ent;
	size_t pos, count, entry_size;
	uint64_t *reg;
	uint64_t prev_tstamp, old_reg;
	
	/* check  for already processed buffer */
	if (hdr->hdr_overflows == last_overflow && hdr->hdr_count == last_count)
		return 0;

	count = hdr->hdr_count;

	ent   = (smpl_entry_t *)(hdr+1);
	pos   = (unsigned long)ent;

	entry_size = sizeof(*ent) + 1 * sizeof(uint64_t);

	prev_tstamp = old_reg = 0;

	while(count--) {
		if (ent->tgid != pid) {
			error_code = 1;
			return -1;
		}

		if (ent->pid != tid) {
			error_code = 2;
			return -1;
		}

		if (ent->last_reset_val != -SMPL_PERIOD) {
			error_code = 3;
			return -1;
		}

		if (ent->ovfl_pmd != ovfl_pmd) {
			error_code = 4;
			return -1;
		}

		if (ent->set != 0) {
			error_code = 5;
			return -1;
		}
		if (ent->tstamp <= prev_tstamp) {
			error_code = 6;
			return -1;
		}
		prev_tstamp = ent->tstamp;
		/* should also test tstamp: monotonically increasing */

		/*
		 * print body: additional PMDs recorded
		 * PMD are recorded in increasing index order
		 */
		reg = (uint64_t *)(ent+1);

		if (*reg < old_reg) {
			error_code = 7;
			return -1;
		}
		old_reg = *reg;

		pos += entry_size;
		ent = (smpl_entry_t *)pos;
		num_entries++;
	}
	return 0;
}

static void
sigio_handler(int n, struct siginfo *info, struct sigcontext *sc)
{
	pfarg_msg_t msg;
	int fd = ctx_fd;
	ssize_t r;
	int ret;

	if (fd != ctx_fd) {
		error_code = 8;
		longjmp(jbuf, 1);
	}

retry:
	r = read(fd, &msg, sizeof(msg));
	if (r != (ssize_t)sizeof(msg)) {
		if(r == -1 && errno == EINTR) {
			goto retry;
		}
		error_code = 9;
		longjmp(jbuf, 1);
	}

	if (msg.type != PFM_MSG_OVFL) {
		error_code = 10;
		longjmp(jbuf, 1);
	}
	ret = process_smpl_buf(getpid(), getpid(), hdr, ovfl_pmd);
	if (ret)
		longjmp(jbuf, 1);
	/*
	 * And resume monitoring
	 */
	if (pfm_restart(fd) == -1) {
		error_code = 11;
		longjmp(jbuf, 1);
	}
}


static int
do_self_smpl(void)
{
	pfarg_ctx_t ctx;
	smpl_arg_t buf_arg;
	pfmlib_input_param_t inp;
	pfmlib_output_param_t outp;
	pfarg_pmd_t pd[NUM_PMDS];
	pfarg_pmc_t pc[NUM_PMCS];
	pfarg_load_t load_args;
	void *buf_addr;
	int ret, fd;
	unsigned int i, num_counters;
	struct sigaction act, oldsig;

	/*
	 * intialize all locals
	 */
	memset(&ctx, 0, sizeof(ctx));
	memset(&buf_arg, 0, sizeof(buf_arg));
	memset(&inp,0, sizeof(inp));
	memset(&outp,0, sizeof(outp));
	memset(pd, 0, sizeof(pd));
	memset(pc, 0, sizeof(pc));
	memset(&load_args, 0, sizeof(load_args));

	ret = pfm_get_num_counters(&num_counters);
	if (ret != PFMLIB_SUCCESS) {
		PFM_LOG("cannot get number of coutners: %s", pfm_strerror(ret));
		return -1;
	}

	ret = pfm_get_cycle_event(&inp.pfp_events[0]);
	if (ret != PFMLIB_SUCCESS) {
		PFM_LOG("cannot find cycle event: %s", pfm_strerror(ret));
		return -1;
	}

	ret = pfm_get_inst_retired_event(&inp.pfp_events[1]);
	if (ret != PFMLIB_SUCCESS) {
		PFM_LOG("cannot find inst retired event: %s", pfm_strerror(ret));
		return -1;
	}

	i = 2;

	/*
	 * set the privilege mode:
	 * 	PFM_PLM3 : user level
	 * 	PFM_PLM0 : kernel level
	 */
	inp.pfp_dfl_plm   = PFM_PLM3;

	if (i > num_counters) {
		i = num_counters;
		PFM_LOG("too many events provided (max=%d events), using first %d event(s)", num_counters, i);
		return -1;
	}
	inp.pfp_event_count = i;

	detect_unavail_pmcs(-1, &inp.pfp_unavail_pmcs);

	if ((ret=pfm_dispatch_events(&inp, NULL, &outp, NULL)) != PFMLIB_SUCCESS) {
		PFM_LOG("cannot configure events: %s", pfm_strerror(ret));
		return -1;
	}

	for (i=0; i < outp.pfp_pmc_count; i++) {
		pc[i].reg_num   = outp.pfp_pmcs[i].reg_num;
		pc[i].reg_value = outp.pfp_pmcs[i].reg_value;
	}

	for (i=0; i < outp.pfp_pmd_count; i++) {
		pd[i].reg_num = outp.pfp_pmds[i].reg_num;

		/*
		 * skip first counter (sampling period)
		 */
		if (i) {
			pfm_bv_set(pd[0].reg_smpl_pmds, pd[i].reg_num);
		}
	}

	pd[0].reg_flags	|= PFM_REGFL_OVFL_NOTIFY;

	/* set sampling periods */
	pd[0].reg_value       = - SMPL_PERIOD;
	pd[0].reg_short_reset = - SMPL_PERIOD;
	pd[0].reg_long_reset  = - SMPL_PERIOD;

	ovfl_pmd = pd[0].reg_num;

	buf_arg.buf_size = 2*getpagesize();

	fd = ctx_fd = pfm_create_context(&ctx, FMT_NAME, &buf_arg, sizeof(buf_arg));
	if (fd == -1) {
		PFM_LOG("cannot create context: %s", strerror(errno));
		return -1;
	}

	/*
	 * retrieve the virtual address at which the sampling
	 * buffer has been mapped
	 */
	buf_addr = mmap(NULL, (size_t)buf_arg.buf_size, PROT_READ, MAP_PRIVATE, fd, 0);
	if (buf_addr == MAP_FAILED) {
		PFM_LOG("cannot mmap sampling buffer: %s", strerror(errno));
		goto error;
	}

	hdr = (smpl_hdr_t *)buf_addr;

	if (pfm_write_pmcs(fd, pc, outp.pfp_pmc_count)) {
		PFM_LOG("pfm_write_pmcs error errno: %s", strerror(errno));
		goto error;
	}
	/*
	 * initialize the PMDs
	 * To be read, each PMD must be either written or declared
	 * as being part of a sample (reg_smpl_pmds, reg_reset_pmds)
	 */
	if (pfm_write_pmds(fd, pd, outp.pfp_pmd_count)) {
		PFM_LOG("pfm_write_pmds error errno: %s", strerror(errno));
		goto error;
	}

	load_args.load_pid = getpid();
	if (pfm_load_context (fd, &load_args)) {
		PFM_LOG("pfm_load_context error: %s", strerror(errno));
		goto error;
	}

	ret = fcntl(fd, F_SETFL, fcntl(fd, F_GETFL, 0) | O_ASYNC);
	if (ret == -1) {
		PFM_LOG("cannot set ASYNC: %s", strerror(errno));
		goto error;
	}

	ret = fcntl(fd, F_SETOWN, getpid());
	if (ret == -1) {
		PFM_LOG("cannot setown: %s", strerror(errno));
		goto error;
	}

	memset(&act, 0, sizeof(act));
	act.sa_handler = (sig_t)sigio_handler;
	sigaction (SIGIO, &act, &oldsig);

	if (sigsetjmp(jbuf, 1)) {
		PFM_LOG("error occurred while processing signal, error_code = %d (check source)", error_code);
		goto error;
	}

	if (pfm_start(fd, NULL)) {
		PFM_LOG("pfm_start error: %s", strerror(errno));
		goto error;
	}
	noploop();

	if (pfm_stop(fd)) {
		PFM_LOG("pfm_stop error: %s", strerror(errno));
		goto error;
	}

	close(fd);

	if (!num_entries) {
		PFM_LOG("did not capture any sample");
		goto error;
	}

	ret = munmap(hdr, (size_t)buf_arg.buf_size);
	if (ret) {
		PFM_LOG("cannot unmap buffer: %s", strerror(errno));
	}
	return ret;
error:
	sigaction(SIGIO, &oldsig, NULL);
	if (hdr)
		munmap(hdr, (size_t)buf_arg.buf_size);
	close(fd);
	return -1;
}

int
main_self_smpl(int argc, char **argv)
{
	int ret;

	/* need library for events */
	if (pfmlib_ok == 0)
		return -2;

	printf("test_self_smpl.test0  "); fflush(stdout);
	ret = do_self_smpl();
	printf("[%s]\n", ret == -1 ? "FAIL" : ret == -2 ? "SKIP" : "PASS");
	return ret;
}
