/*
 * test_read_pmcd.c - test pfm_read_pmds
 *
 * Copyright (c) 2008 Google, Inc
 * Contributed by Stephane Eranian <eranian@google.com>
 */
#include <sys/types.h>
#include <inttypes.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <sys/wait.h>
#include <sys/mman.h>

#include <perfmon/perfmon.h>
#include <perfmon/perfmon_dfl_smpl.h>
#include <perfmon/pfmlib.h>

#include "pfm_tests.h"
#include "detect_pmcs.h"

/*
 * return first available register index from unavailable register bitmask
 */
static int
get_avail_reg(pfmlib_regmask_t *unavail)
{
	int i;
	/* look for first available register */
	for (i=0; i < PFMLIB_REG_MAX; i++) {
		if (!pfm_regmask_isset(unavail, i))
			break;
	}
	return i == PFMLIB_REG_MAX ? -1 : i;
}

/*
 * test NULL register vector and size are rejected
 */
static int
test1(int fd, pfmlib_regmask_t *unavail)
{
	int ret;

	/* null vector, null size */
	ret = pfm_read_pmds(fd, NULL, 0);
	if (ret == -1) {
		PFM_LOG("unexpected error: %s", strerror(errno));
		return -1;
	}
	return 0;
}

/*
 * test NULL vector is rejected
 */
static int
test2(int fd, pfmlib_regmask_t *unavail)
{
	int ret;

	/* null vector */
	ret = pfm_read_pmds(fd, NULL, sizeof(pfarg_pmd_t));
	if (ret == 0) {
		PFM_LOG("should not have accepted write pmds");
		return -1;
	}
	return 0;
}


/*
 * test non NULL vector but zero size is accepted: do nothing
 */
static int
test3(int fd, pfmlib_regmask_t *unavail)
{
	int ret;
	pfarg_pmd_t pd[1];

	memset(pd, 0, sizeof(pd));

	/* null size */
	ret = pfm_read_pmds(fd, pd, 0);
	if (ret == -1) {
		PFM_LOG("unexpected error: %s", strerror(errno));
		return -1;
	}
	return 0;
}

/*
 * test file descriptor is valid, i.e, perfmon related
 */
static int
test5(int fd, pfmlib_regmask_t *unavail)
{
	int ret;
	pfarg_pmd_t pd[1];

	memset(pd, 0, sizeof(pd));

	/* use a non perfmon-related fd */
	ret = pfm_read_pmds(0, pd, 1);
	if (ret == 0) {
		PFM_LOG("should not have accepted pmcs write");
		return -1;
	}
	if (errno != EBADF) {
		PFM_LOG("unexpected errno=%d vs EBADF", errno);
		return -1;
	}
	return 0;
}

/*
 * test vector is too large for syscall
 */
static int
test6(int fd, pfmlib_regmask_t *unavail)
{
	int i, ret, avail;
	pfarg_pmd_t *pd;
	unsigned long max;
	size_t n;
	char *str;

	str = read_sysfs("/sys/kernel/perfmon/arg_mem_max");
	if (!str) {
		PFM_LOG("cannot read arg_mem_max");
		return -1;
	}

	avail = get_avail_reg(unavail);
	if (avail == -1) {
		PFM_LOG("no available registers");
		return -1;
	}

	max = strtoul(str, NULL, 0);
	free(str);
	n = max / sizeof(pfarg_pmd_t);
	n++;

	pd = calloc(n, sizeof(pfarg_pmd_t));
	if (!pd) {
		PFM_LOG("failed to allocate space for %zu pfarg_pmd_t",n);
		free(str);
		return -1;
	}

	memset(pd, 0, n * sizeof(pfarg_pmd_t));
	/* make sure we use a valid register */
	for (i=0; i < n ; i++)
		pd[i].reg_num = avail;

	/* must write register before it can be read */
	ret = pfm_write_pmds(fd, pd, 1);
	if (ret) {
		PFM_LOG("failed to write pmd: %s", strerror(errno));
		free(pd);
		return -1;
	}

	/* vector is too large */
	ret = pfm_write_pmds(fd, pd, n);
	free(pd);
	if (ret == -1) {
		if (errno != E2BIG) {
			PFM_LOG("unexpected errno: %s", strerror(errno));
			return -1;
		}
		return 0;
	}
	PFM_LOG("should not have accepted pmds write");
	return -1;
}

/*
 * test reading of unavailable registers
 */
static int
test7(int fd, pfmlib_regmask_t *unavail)
{
	pfarg_pmd_t pd[1];
	int i, ret;

	/* look for unimplemented/unavailable register */
	for (i=0; i < PFMLIB_REG_MAX; i++) {
		if (pfm_regmask_isset(unavail, i))
			goto found;
	}
	PFM_LOG("cannot find unimplemented registers, skipping test");
	return 0;
found:
	memset(pd, 0, sizeof(pd));
	pd[0].reg_num = i;

	/* try reading unavailable pmd */
	ret = pfm_read_pmds(fd, pd, 1);
	if (ret == -1) {
		if (errno != EINVAL) {
			PFM_LOG("unexpected errno: %s", strerror(errno));
			return -1;
		}
		return 0;
	}
	PFM_LOG("should not have accepted pmd read");
	return -1;
}

/*
 * test invalid vector, i.e., unexisting page
 */
static int
test8(int fd, pfmlib_regmask_t *unavail)
{
	pfarg_pmd_t *pd;
	size_t pgsz;
	void *addr;
	int ret, avail;

	avail = get_avail_reg(unavail);
	if (avail == -1) {
		PFM_LOG("no available registers");
		return -1;
	}
	pgsz = getpagesize();

	addr = mmap(NULL, 2 * pgsz, PROT_READ|PROT_WRITE, MAP_PRIVATE|MAP_ANONYMOUS, -1, 0);
	if (addr == MAP_FAILED) {
		PFM_LOG("mmap failed: %s", strerror(errno));
		return -1;
	}
	ret = munmap(addr + pgsz, pgsz);
	if (ret == -1) {
		PFM_LOG("munmap failed: %s", strerror(errno));
		return -1;
	}
	pd = addr +pgsz - sizeof(*pd);

	memset(pd, 0, sizeof(pd));
	pd[0].reg_num = avail;

	/* must write register before it can be read */
	ret = pfm_write_pmds(fd, pd, 1);
	if (ret) {
		PFM_LOG("failed to write pmd: %s", strerror(errno));
		return -1;
	}

	/* pass invalid memory (2nd element of vector) */
	ret = pfm_read_pmds(fd, pd, 2);
	if (ret == 0) {
		PFM_LOG("should not have accepted read pmds");
		munmap(addr, pgsz);
		return -1;
	}
	if (ret == -1 && errno != EFAULT) {
		PFM_LOG("unexpected errno=%d vs. EFAULT", errno);
	} else
		ret = 0;
	munmap(addr, pgsz);
	return ret;
}

/*
 * test pmd attribute read
 */
static int
test9(int fd, pfmlib_regmask_t *unavail)
{
	pfmlib_input_param_t inp;
	pfmlib_output_param_t outp;
	pfarg_pmc_t pc[8];
	pfarg_pmd_t pd[8];
	int i, ret;
	
	/* ignore if libpfm not initialized */
	if (pfmlib_ok == 0)
		return -2;

	memset(&inp,0, sizeof(inp));
	memset(&outp,0, sizeof(outp));
	memset(pc, 0, sizeof(pc));
	memset(pd, 0, sizeof(pd));

	ret = pfm_get_cycle_event(&inp.pfp_events[0]);
	if (ret != PFMLIB_SUCCESS) {
		PFM_LOG("cannot find cycle event: %s", pfm_strerror(ret));
		return -1;
	}
	inp.pfp_dfl_plm = PFM_PLM3;
	inp.pfp_event_count = 1;
	inp.pfp_unavail_pmcs = *unavail;

	ret = pfm_dispatch_events(&inp, NULL, &outp, NULL);
	if (ret != PFMLIB_SUCCESS) {
		PFM_LOG("failed dispatch_events: %s", pfm_strerror(ret));
		return -1;
	}

	if (outp.pfp_pmc_count == 0) {
		PFM_LOG("errno no pmc to write");
		return -1;
	}

	if (outp.pfp_pmc_count*sizeof(pfarg_pmc_t) > sizeof(pc)) {
		PFM_LOG("pc vector is too small");
		return -1;
	}

	if (outp.pfp_pmd_count*sizeof(pfarg_pmd_t) > sizeof(pd)) {
		PFM_LOG("pd vector is too small");
		return -1;
	}

	for (i=0; i < outp.pfp_pmc_count; i++) {
		pc[i].reg_num   = outp.pfp_pmcs[i].reg_num;
		pc[i].reg_value = outp.pfp_pmcs[i].reg_value;
	}

	for (i=0; i < outp.pfp_pmd_count; i++) {
		pd[i].reg_num   = outp.pfp_pmds[i].reg_num;
		pd[i].reg_value = i;
		pd[i].reg_short_reset = i+1;
		pd[i].reg_long_reset = i+2;
		pd[i].reg_random_mask = i+3;
		pd[i].reg_ovfl_switch_cnt = i+4;
	}

	if (pfm_write_pmds(fd, pd, outp.pfp_pmd_count)) {
		PFM_LOG("cannot write pmd: %s", strerror(errno));
		return -1;
	}

	/* clear all fields but reg_num to test read */
	for (i=0; i < outp.pfp_pmd_count; i++) {
		pd[i].reg_value = 0;
		pd[i].reg_short_reset = 0;
		pd[i].reg_long_reset = 0;
		pd[i].reg_random_mask = 0;
		pd[i].reg_ovfl_switch_cnt = 0;
	}

	if (pfm_read_pmds(fd, pd, outp.pfp_pmd_count)) {
		PFM_LOG("cannot read pmd: %s", strerror(errno));
		return -1;
	}

	for(i=0; i < outp.pfp_pmd_count; i++) {
		if (pd[i].reg_value != i) {
			PFM_LOG("unexpected pmd value = 0x%"PRIu64, pd[0].reg_value);
			return -1;
		}

		if (pd[i].reg_last_reset_val != 0) {
			PFM_LOG("unexpected pmd last reset = 0x%"PRIu64, pd[0].reg_last_reset_val);
			return -1;
		}
		if (pd[i].reg_ovfl_switch_cnt != i+4) {
			PFM_LOG("unexpected pmd switch cnt= 0x%"PRIu64, pd[0].reg_ovfl_switch_cnt);
			return -1;
		}

		/*
		 * following fields are set by user and not changed by kernel, therefore they are not
		 * updated on read
		 */
		if (pd[i].reg_short_reset != 0) {
			PFM_LOG("unexpected pmd short reset = 0x%"PRIu64, pd[0].reg_short_reset);
			return -1;
		}

		if (pd[i].reg_long_reset != 0) {
			PFM_LOG("unexpected pmd long reset = 0x%"PRIu64, pd[0].reg_long_reset);
			return -1;
		}

		if (pd[i].reg_random_mask != 0) {
			PFM_LOG("unexpected pmd random mask= 0x%"PRIu64, pd[0].reg_random_mask);
			return -1;
		}
	}
	return 0;
}

/*
 * test that we must write to a pmd before being allowed to read it
 */
static int
test10(int fd, pfmlib_regmask_t *unavail)
{
	pfarg_ctx_t ctx;
	pfarg_pmd_t pd[1];
	int ret, avail, fd2;

	avail = get_avail_reg(unavail);
	if (avail == -1) {
		PFM_LOG("no available registers");
		return -1;
	}

	memset(&ctx,0, sizeof(ctx));
	memset(pd, 0, sizeof(pd));

	/*
	 * need to create a new context to avoid register access pollution
	 * from other tests
	 */
	fd2 = pfm_create_context(&ctx, NULL, NULL, 0);
	if (fd2 == -1) {
		PFM_LOG("canot create context: %s", strerror(errno));
		return -1;
	}
	pd[0].reg_num = avail;

	/* cannot read a register directly, must write to it first */
	ret = pfm_read_pmds(fd2, pd, 1);
	if (ret == 0) {
		PFM_LOG("cannot read pmd: %s", strerror(errno));
		close(fd2);
		return -1;
	}
	close(fd2);

	return 0;
}

/*
 * test that we can read pmd that have been specified as smpl_pmd
 */
static int
test11(int fd, pfmlib_regmask_t *unavail)
{
	pfarg_ctx_t ctx;
	pfarg_pmd_t pd[1];
	int i, ret, avail[2], fd2, n;

	for (i=n=0; i < PFMLIB_REG_MAX; i++) {
		if (!pfm_regmask_isset(unavail, i)) {
			avail[n++] = i;
			if (n == 2)
				break;
		}
	}
	/* ingore if no 2 counters are available */
	if (n != 2) {
		PFM_LOG("could not find 2 available registers");
		return -2;
	}

	memset(&ctx,0, sizeof(ctx));
	memset(pd, 0, sizeof(pd));

	/*
	 * need to create a new context to avoid register access pollution from other
	 * tests
	 */
	fd2 = pfm_create_context(&ctx, NULL, NULL, 0);
	if (fd2 == -1) {
		PFM_LOG("canot create context: %s", strerror(errno));
		return -1;
	}
	pd[0].reg_num = avail[0];
	pfm_bv_set(pd[0].reg_smpl_pmds, avail[1]);

	ret = pfm_write_pmds(fd2, pd, 1);
	if (ret == -1) {
		PFM_LOG("cannot write pmd: %s", strerror(errno));
		close(fd2);
		return -1;
	}
	pd[0].reg_num = avail[1];

	/* can read 2nd pmd because was consider accessed via smpl_pmds */
	ret = pfm_read_pmds(fd2, pd, 1);
	if (ret == -1) {
		PFM_LOG("cannot read pmd: %s", strerror(errno));
	}
	close(fd2);

	return ret;
}

/*
 * test that we can read pmd that have been specified as reset_pmd
 */
static int
test12(int fd, pfmlib_regmask_t *unavail)
{
	pfarg_ctx_t ctx;
	pfarg_pmd_t pd[1];
	pfmlib_regmask_t impl_cnts;
	int i, ret, avail[2], fd2, n;

	/* ignore if libpfm is not initialized */
	if (pfmlib_ok == 0)
		return -2;

	pfm_get_impl_counters(&impl_cnts);

	for (i=n=0; i < PFMLIB_REG_MAX; i++) {
		if (!pfm_regmask_isset(unavail, i) && pfm_regmask_isset(&impl_cnts, i)) {
			avail[n++] = i;
			if (n == 2)
				break;
		}
	}
	/* ignreo if no 2 counters are available */
	if (n != 2) {
		PFM_LOG("could not find 2 available registers");
		return -2;
	}

	memset(&ctx,0, sizeof(ctx));
	memset(pd, 0, sizeof(pd));

	/*
	 * need to create a new context to avoid register access pollution from other
	 * tests
	 */
	fd2 = pfm_create_context(&ctx, NULL, NULL, 0);
	if (fd2 == -1) {
		PFM_LOG("canot create context: %s", strerror(errno));
		return -1;
	}
	pd[0].reg_num = avail[0];
	pfm_bv_set(pd[0].reg_reset_pmds, avail[1]);

	ret = pfm_write_pmds(fd2, pd, 1);
	if (ret == -1) {
		PFM_LOG("cannot write pmd: %s", strerror(errno));
		close(fd2);
		return -1;
	}
	pd[0].reg_num = avail[1];

	/* can read 2nd pmd because was considered accessed via reset_pmds */
	ret = pfm_read_pmds(fd2, pd, 1);
	if (ret == -1) {
		PFM_LOG("cannot read pmd: %s", strerror(errno));
	}
	close(fd2);

	return ret;
}

/*
 * test pmd reading in child process while unloaded
 */
static int
test13(int fd, pfmlib_regmask_t *unavail)
{
	pfarg_pmd_t pd[1];
	int ret, avail;
	int status;
	uint64_t val;
	pid_t pid;

	avail = get_avail_reg(unavail);
	if (avail == -1) {
		PFM_LOG("no available registers");
		return -1;
	}

	memset(pd, 0, sizeof(pd));

	pd[0].reg_num = avail;
	pd[0].reg_value  = val = 0x123456789ull;

	ret = pfm_write_pmds(fd, pd, 1);
	if (ret == -1) {
		PFM_LOG("cannot write pmd: %s", strerror(errno));
		goto error;
	}

	pd[0].reg_value  = 0;

	ret = pfm_read_pmds(fd, pd, 1);
	if (ret == -1) {
		PFM_LOG("cannot read pmd: %s", strerror(errno));
		goto error;
	}
	if (pd[0].reg_value != val) {
		PFM_LOG("invalid pmd value %"PRIu64, pd[0].reg_value);
		goto error;
	}
	pid = fork();
	if (pid == -1) {
		PFM_LOG("cannot fork: %s", strerror(errno));
		goto error;
	}

	if (pid == 0) {
		ret = pfm_read_pmds(fd, pd, 1);
		if (ret == -1) {
			PFM_LOG("cannot read pmd: %s", strerror(errno));
			goto error;
		}
		if (pd[0].reg_value != val) {
			PFM_LOG("invalid pmd value %"PRIu64, pd[0].reg_value);
			exit(-1);
		}
		exit(0);
	}
	ret = waitpid(pid, &status, 0);
	if (ret == -1) {
		PFM_LOG("waitpid failed: %s", strerror(errno));
		goto error;
	}
	return WEXITSTATUS(status) ? -1 : 0;
error:	
	return -1;
}

static int (*ctx_tests[])(int fd, pfmlib_regmask_t *unavail)={
	test1,
	test2,
	test3,
	/* test4 is empty */
	test5,
	test6,
	test7,
	test8,
	test9,
	test10,
	test11,
	test12,
	test13,
	NULL
};

int
main_pfm_read_pmds(int argc, char **argv)
{
	pfarg_ctx_t ctx;
	pfmlib_regmask_t unavail_pmcs, unavail_pmds;
	int ctx_fd, ret = -1, i;
	int (**pf)(int fd, pfmlib_regmask_t *unavail);

	/*
	 * ignore if pfmlib initialization failed
	 * this  is needed because we need to figure out which
	 * pmcs are available
	 */
	if (pfmlib_ok == 0)
		return -2;

	memset(&ctx, 0, sizeof(ctx));

	ctx_fd = pfm_create_context(&ctx, NULL, NULL, 0);
	if (ctx_fd == -1)  {
		PFM_LOG("unexpected context creation error: %s", strerror(errno));
		return -1;
	}

	memset(&unavail_pmcs, 0, sizeof(unavail_pmcs));
	memset(&unavail_pmds, 0, sizeof(unavail_pmds));

	/* ignroe return value: best effort */
	detect_unavail_pmu_regs(ctx_fd, &unavail_pmcs, &unavail_pmds);

	for (pf = ctx_tests, i= 0; *pf ; pf++, i++) {
		printf("pfm_read_pmds.test%-3d", i); fflush(stdout);
		ret = (*pf)(ctx_fd, &unavail_pmds);
		printf("[%s]\n", ret == -1 ? "FAIL" : ret == -2 ? "SKIP" : "PASS");
		/* -2 means skip test */
		if (ret == -1)
			break;
	}

	close(ctx_fd);

	return ret;
}
