/*
 * test_write_pmcs.c - test pfm_write_pmcs
 *
 * Copyright (c) 2008 Google, Inc
 * Contributed by Stephane Eranian <eranian@google.com>
 */
#include <sys/types.h>
#include <inttypes.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>
#include <sys/wait.h>
#include <sys/mman.h>

#include <perfmon/perfmon.h>
#include <perfmon/perfmon_dfl_smpl.h>
#include <perfmon/pfmlib.h>

#include "pfm_tests.h"
#include "detect_pmcs.h"

static int
get_avail_reg(pfmlib_regmask_t *unavail)
{
	int i;
	/* look for first available register */
	for (i=0; i < PFMLIB_REG_MAX; i++) {
		if (!pfm_regmask_isset(unavail, i))
			break;
	}
	return i == PFMLIB_REG_MAX ? -1 : i;
}

static int
test1(int fd, pfmlib_regmask_t *unavail)
{
	int ret;

	/* null vector, null size */
	ret = pfm_write_pmcs(fd, NULL, 0);
	if (ret == -1) {
		PFM_LOG("unexpected error: %s", strerror(errno));
		return -1;
	}
	return 0;
}

static int
test2(int fd, pfmlib_regmask_t *unavail)
{
	int ret;

	/* null vector */
	ret = pfm_write_pmcs(fd, NULL, sizeof(pfarg_pmc_t));
	if (ret == 0) {
		PFM_LOG("should not have accepted write pmcs");
		return -1;
	}
	return 0;
}


static int
test3(int fd, pfmlib_regmask_t *unavail)
{
	int ret;
	pfarg_pmc_t pc[1];

	memset(pc, 0, sizeof(pc));

	/* null size */
	ret = pfm_write_pmcs(fd, pc, 0);
	if (ret == -1) {
		PFM_LOG("unexpected error: %s", strerror(errno));
		return -1;
	}
	return 0;
}

static int
test5(int fd, pfmlib_regmask_t *unavail)
{
	int ret;
	pfarg_pmc_t pc[1];

	memset(pc, 0, sizeof(pc));

	/* use a non perfmon-related fd */
	ret = pfm_write_pmcs(0, pc, 1);
	if (ret == 0) {
		PFM_LOG("should not have accepted pmcs write");
		return -1;
	}
	if (errno != EBADF) {
		PFM_LOG("unexpected errno=%d vs EBADF", errno);
		return -1;
	}
	return 0;
}

static int
test6(int fd, pfmlib_regmask_t *unavail)
{
	int i, ret, avail;
	pfarg_pmc_t *pc;
	unsigned long max;
	size_t n;
	char *str;

	str = read_sysfs("/sys/kernel/perfmon/arg_mem_max");
	if (!str) {
		PFM_LOG("cannot read arg_mem_max");
		return -1;
	}
	avail = get_avail_reg(unavail);
	if (avail == -1) {
		PFM_LOG("no available registers");
		return -1;
	}

	max = strtoul(str, NULL, 0);
	free(str);
	n = max / sizeof(pfarg_pmc_t);
	n++;

	pc = calloc(n, sizeof(pfarg_pmc_t));
	if (!pc) {
		PFM_LOG("failed to allocate space for %zu pfarg_pmc_t",n);
		free(str);
		return -1;
	}

	memset(pc, 0, n * sizeof(pfarg_pmc_t));

	/* make sure we use a valid register */
	for (i=0; i < n ; i++)
		pc[i].reg_num = avail;

	/* vector is too large */
	ret = pfm_write_pmcs(fd, pc, n);
	free(pc);
	if (ret == -1) {
		if (errno != E2BIG) {
			PFM_LOG("unexpected errno: %s", strerror(errno));
			return -1;
		}
		return 0;
	}
	PFM_LOG("should not have accepted pmcs write");
	return -1;
}

static int
test7(int fd, pfmlib_regmask_t *unavail)
{
	pfarg_pmc_t pc[1];
	int i, ret;

	/* look for unimplemented/unavailable register */
	for (i=0; i < PFMLIB_REG_MAX; i++) {
		if (pfm_regmask_isset(unavail, i))
			goto found;
	}
	PFM_LOG("cannot find unimplemented registers, skipping test");
	return -2;
found:
	memset(pc, 0, sizeof(pc));
	pc[0].reg_num = i;

	/* try writing unavailable pmc */
	ret = pfm_write_pmcs(fd, pc, 1);
	if (ret == -1) {
		if (errno != EINVAL) {
			PFM_LOG("unexpected errno: %s", strerror(errno));
			return -1;
		}
		return 0;
	}
	PFM_LOG("should not have accepted pmc write");
	return -1;
}

static int
test8(int fd, pfmlib_regmask_t *unavail)
{
	pfarg_pmc_t *pc;
	size_t pgsz;
	void *addr;
	int ret, avail;

	avail = get_avail_reg(unavail);
	if (avail == -1) {
		PFM_LOG("cannot find available register");
		return -1;
	}

	pgsz = getpagesize();

	addr = mmap(NULL, 2 * pgsz, PROT_READ|PROT_WRITE, MAP_PRIVATE|MAP_ANONYMOUS, -1, 0);
	if (addr == MAP_FAILED) {
		PFM_LOG("mmap failed: %s", strerror(errno));
		return -1;
	}
	ret = munmap(addr + pgsz, pgsz);
	if (ret == -1) {
		PFM_LOG("munmap failed: %s", strerror(errno));
		return -1;
	}
	pc = addr +pgsz - sizeof(*pc);

	memset(pc, 0, sizeof(pc));
	/* make sure we use a valid regiser */
	pc[0].reg_num = avail;

	ret = pfm_write_pmcs(fd, pc, 1);
	if (ret) {
		PFM_LOG("unexpected error: %s", strerror(errno));
		munmap(addr, pgsz);
		return -1;
	}
	
	/* pass invalid memory (2nd element of vector) */
	ret = pfm_write_pmcs(fd, pc, 2);
	if (ret == 0) {
		PFM_LOG("should not have accepted write pmcs");
		munmap(addr, pgsz);
		return -1;
	}

	if (ret == -1 && errno != EFAULT) {
		PFM_LOG("unexpected errno=%d vs. EFAULT", errno);
	} else
		ret = 0;

	munmap(addr, pgsz);

	return ret;
}

static int
test9(int fd, pfmlib_regmask_t *unavail)
{
	pfmlib_input_param_t inp;
	pfmlib_output_param_t outp;
	pfarg_pmc_t pc[8];
	int i, ret;
	
	memset(&inp,0, sizeof(inp));
	memset(&outp,0, sizeof(outp));
	memset(pc,0, sizeof(pc));

	ret = pfm_get_cycle_event(&inp.pfp_events[0]);
	if (ret != PFMLIB_SUCCESS) {
		PFM_LOG("cannot find cycle event: %s", pfm_strerror(ret));
		return -1;
	}
	inp.pfp_dfl_plm = PFM_PLM3;
	inp.pfp_event_count = 1;
	inp.pfp_unavail_pmcs = *unavail;

	ret = pfm_dispatch_events(&inp, NULL, &outp, NULL);
	if (ret != PFMLIB_SUCCESS) {
		PFM_LOG("failed dispatch_events: %s", pfm_strerror(ret));
		return -1;
	}

	if (outp.pfp_pmc_count == 0) {
		PFM_LOG("errno no pmc to write");
		return -1;
	}

	if (outp.pfp_pmc_count*sizeof(pfarg_pmc_t) > sizeof(pc)) {
		PFM_LOG("pc vector is too small");
		return -1;
	}

	for (i=0; i < outp.pfp_pmc_count; i++) {
		pc[i].reg_num   = outp.pfp_pmcs[i].reg_num;
		pc[i].reg_value = outp.pfp_pmcs[i].reg_value;
	}

	if (pfm_write_pmcs(fd, pc, outp.pfp_pmc_count)) {
		PFM_LOG("cannot write pmc: %s", strerror(errno));
		return -1;
	}

#if 0
	/* this test causes problem on Intel processor with fixed counters
	 * which do support EMUL64. Yet cycle event uses a fixed counter
	 */
	/* check NO_EMUL64 */
	pc[0].reg_flags = PFM_REGFL_NO_EMUL64;

	if (pfm_write_pmcs(fd, pc, 1)) {
		PFM_LOG("cannot write pmc: %s", strerror(errno));
		return -1;
	}
#endif
	return 0;
}

/*
 * tests of pfm_write_pmcs() after pfm_load_context() are deferred to
 * a series after the tests for pfm_load_context
 */

static int (*ctx_tests[])(int fd, pfmlib_regmask_t *unavail)={
	test1,
	test2,
	test3,
	/* test 4 is empty */
	test5,
	test6,
	test7,
	test8,
	test9,
	NULL
};

int
main_pfm_write_pmcs(int argc, char **argv)
{
	pfarg_ctx_t ctx;
	pfmlib_regmask_t unavail;
	int ctx_fd, ret = -1, i;
	int (**pf)(int fd, pfmlib_regmask_t *unavail);

	memset(&ctx, 0, sizeof(ctx));

	ctx_fd = pfm_create_context(&ctx, NULL, NULL, 0);
	if (ctx_fd == -1)  {
		PFM_LOG("unexpected context creation error: %s", strerror(errno));
		return -1;
	}

	memset(&unavail, 0, sizeof(unavail));

	/* ignroe return value: best effort */
	detect_unavail_pmcs(ctx_fd, &unavail);

	for (pf = ctx_tests, i=0; *pf ; pf++, i++) {
		printf("pfm_write_pmcs.test%-3d ", i); fflush(stdout);
		ret = (*pf)(ctx_fd, &unavail);
		printf("[%s]\n", ret == -1 ? "FAIL" : ret == -2 ? "SKIP" : "PASS");
		/* -2 means skipped */
		if (ret == -1)
			break;
	}

	close(ctx_fd);

	return ret;
}
