#define _XOPEN_SOURCE 600
#define _GNU_SOURCE

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <sys/fcntl.h>
#include <sys/mman.h>
#include <sys/wait.h>

#define BIGSIZE 128*1024*1024
#define READSIZE 32*1024*1024
#define WRITESIZE 32*1024*1024

int pagesize;
char *iobuf;

char *pagebuf;

void assert(const char *what, int assertion)
{
	if (assertion)
		return;
	perror(what);
	exit(1);
}

void do_buffered_writes(int fd, int pattern)
{
	int rc;
	int offset;
	
	memset(iobuf, pattern, WRITESIZE);
	for (offset = 0; offset+WRITESIZE <= BIGSIZE; offset += WRITESIZE) {
		rc = pwrite(fd, iobuf, WRITESIZE, offset);
		assert("pwrite", rc >= 0);
		if (rc != WRITESIZE) {
			fprintf(stderr, "short write (%d out of %d)\n",
				rc, WRITESIZE);
			exit(1);
		}
		fsync(fd);
	}
}

int do_direct_reads(char *filename)
{
	int fd;
	int offset;
	int rc, i;
	int *p;
	
	fd = open(filename, O_DIRECT|O_RDONLY, 0);
	assert("open", fd >= 0);

	for (offset = 0; offset+READSIZE <= BIGSIZE; offset += READSIZE) {
		rc = pread(fd, iobuf, READSIZE, offset);
		assert("pread", rc >= 0);
		if (rc != READSIZE) {
			fprintf(stderr, "short read (%d out of %d)\n",
				rc, READSIZE);
			exit(1);
		}
		for (i=0, p=(int *)iobuf; i<READSIZE; i+=4) {
			int *p2;
			int j;
			if (*p) {
				char cmd[1024];
				int fd2;
				int readdata2;

				fprintf(stderr,
					"Found data (%08x) at offset %d+%d\n",
					*p, offset, i);

				/*
				 * check how much is non-zero.
				 */
				for (p2 = p + 1, j = i + 4; j < READSIZE; j += 4) {

					if (*p2 == 0)
						break;
					p2++;
				}
				fprintf(stderr,
					"Found data (%08x) to offset %d+%d\n",
					*(p2 - 1), offset, j - 1);
				/*
				 * read the data buffered interface to check.
				 */
				fd2 = open(filename, O_RDONLY, 0);
				pread(fd2, &readdata2, sizeof(readdata2),
					offset+i);
				fprintf(stderr,
				"Buffered data (%08x) at offset %d+%d\n",
					readdata2, offset, i);

				/*
				 * Re-read using DIO to read page with
				 * bad data.
				 */
				pread(fd, pagebuf, pagesize, 
					(offset + i) & ~(pagesize - 1));
				fprintf(stderr,
					"DIO read2 data (%08x) %d+%d\n",
			*(int *)&pagebuf[((offset + i) & (pagesize - 1))],
					(offset + i) & ~(pagesize - 1),
					(offset + i) & (pagesize - 1));
				close(fd2);
				

				/*
				 * more debug output
				 */
				sprintf(cmd, "od -c %s > %s.od.output",
					filename, filename);
				system(cmd);
				
				close(fd);
				return 1;
			}
			p++;
		}
	}
	close(fd);
	return 0;
}

int main(int argc, char *argv[])
{
	char *filename;
	int fd;
	int pid;
	int err;
	int pass = 0;
	int bufsize;
	
	if (argc != 2) {
		fprintf(stderr, "Needs a filename as an argument.\n");
		exit(1);
	}
	
	filename = argv[1];
	
	pagesize = getpagesize();
	bufsize = READSIZE;
	if (WRITESIZE > READSIZE)
		bufsize = WRITESIZE;
	err = posix_memalign((void**) &iobuf, pagesize, bufsize);
	err |= posix_memalign((void**) &pagebuf, pagesize, pagesize);
	if (err) {
		fprintf(stderr, "Error allocating %d aligned bytes.\n", bufsize);
		exit(1);
	}
	
	fd = open(filename, O_CREAT|O_TRUNC|O_RDWR, 0666);
	assert("open", fd >= 0);
	
	do {
		printf("Pass %d...\n", ++pass);
		
		assert("ftruncate", ftruncate(fd, BIGSIZE) == 0);
		fsync(fd);

		pid = fork();
		assert("fork", pid >= 0);
		
		if (!pid) {
			do_buffered_writes(fd, 0);
			exit(0);
		}
		
		err = do_direct_reads(filename);

		wait4(pid, NULL, 0, 0);
		
		if (err) 
			break;

		/* Fill the file with a known pattern so that the blocks
		 * on disk can be detected if they become exposed. */
		do_buffered_writes(fd, 1);
		fsync(fd);

		assert("ftruncate", ftruncate(fd, 0) == 0);
		fsync(fd);
	} while (1);
	
	return err;
}

