/*
 * Show file blocks.
 *
 * jeremy@goop.org wrote this.
 */

#include <unistd.h>
#include <stdio.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>

#include <linux/fs.h>

static const char *myname;

#ifdef __x86_64__
typedef unsigned long long blktype;
#else
typedef unsigned long blktype;
#endif

int main(int argc, char **argv)
{
	int i, err = 0;
	int fd;
	blktype blksize = 0;
	off_t filesz;
	blktype fileblks = 0;
	blktype blk;
	blktype start = -1;
	blktype end = -1;
	blktype first_logical = -1;

	myname = argv[0];

	while ((i = getopt(argc, argv, "")) != EOF) {
		switch(i) {
		default:
			err++;
			break;
		}
	}

	if (err || optind != argc-1) {
		fprintf(stderr, "Usage: %s file\n",
			myname);
		exit(1);
	}

	fd = open(argv[optind], O_RDONLY);
	if (fd == -1) {
		perror(argv[optind]);
		exit(1);
	}

	if (ioctl(fd, FIGETBSZ, &blksize) == -1) {
		perror("FIGETBSZ");
		fprintf(stderr, "assuming 4096\n");
		blksize = 4096;
	}

	filesz = lseek(fd, 0, SEEK_END);
	lseek(fd, 0, SEEK_SET);
	fileblks = (filesz + blksize-1) / blksize;

	printf("blksize: %lld, %lld blocks\n", blksize, fileblks);

	err = 0;
	for (blk = 0; blk < fileblks; blk++) {
		blktype devblk = blk;

		if (ioctl(fd, FIBMAP, &devblk) == -1) {
			if (errno == -EPERM) {
				fprintf(stderr, "got root?\n");
				exit(1);
			}
			printf("%llu: %d (%s)\n",
				(unsigned long long)blk,
				errno, strerror(errno));
			err++;
		} else {
			if (start == -1) {
				start = devblk;
				end = devblk;
				first_logical = blk;
			} else {
				if (devblk == end + 1) {
					end++;
				} else {
					printf("%llu-%llu: %llu-%llu (%llu)\n",
						(unsigned long long)first_logical,
						(unsigned long long)(first_logical+(end-start)),
						(unsigned long long)start,
						(unsigned long long)end,
						(unsigned long long)(end - start + 1));
					start = devblk;
					end = devblk;
					first_logical = blk;
				}
			}
		}
	}

	if (start != -1)
		printf("%llu-%llu: %llu-%llu (%llu)\n",
			(unsigned long long)first_logical,
			(unsigned long long)(first_logical+(end-start)),
			(unsigned long long)start,
			(unsigned long long)end,
			(unsigned long long)(end - start + 1));

	exit(err ? 1 : 0);
}
