#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>
#include "slacker.h"

char *ourname;
char _meg[1024*1024 + 40960];
int do_fsync = 0;
unsigned align_offset = 0;
int o_sync;
int blocksize = 0;

static void usage(void)
{
	fprintf(stderr,
		"Usage: %s -m nmegs [-fkoO] [-a align] [-B blocksize] [-c chunksize(bytes)] filename\n"
		"          -a align      : O_DIRECT memory alignment (bytes)\n"
		"          -f            : fsync after write\n"
		"          -k            : -c is in kbytes\n"
		"          -o            : use O_DIRECT\n"
		"          -s            : use O_SYNC\n"
		"          -O            : file overwrite\n",
		ourname);
	exit(1);
}

static void *round_up(void *ptr, unsigned align, unsigned offset)
{
	unsigned long ret = (unsigned long)ptr;

	ret = ((ret + align - 1) & ~(align - 1));
	ret += offset;
	return (void *)ret;
}

static void set_blocksize(int fd)
{
	if (blocksize) {
		if (ioctl(fd, BLKBSZSET, &blocksize) < 0) {
			perror("ioctl");
			exit(1);
		}
	}
}

int main(int argc, char *argv[])
{
	char *filename;
	unsigned long nmegs = 0;
	int fd;
	int chunk = 1024 * 1024;
	char *meg;
	int c;
	int o_direct = 0;
	int overwrite = O_CREAT|O_TRUNC;
	int kilobytes = 0;

	ourname = argv[0];

	if (argc < 2)
		usage();

	while ((c = getopt(argc, argv, "fkoOsa:B:m:c:")) != -1) {
		switch (c) {
		case 'a':
			align_offset = strtol(optarg, NULL, 10);
			break;
		case 'B':
			blocksize = strtol(optarg, NULL, 10);
			break;
		case 'c':
			chunk = strtol(optarg, NULL, 10);
			break;
		case 'f':
			do_fsync++;
			break;
		case 'k':
			kilobytes++;
			break;
		case 'o':
			o_direct = O_DIRECT;
			break;
		case 'O':
			overwrite = 0;
			break;
		case 'm':
			nmegs = strtol(optarg, NULL, 10);
			break;
		case 's':
			o_sync = O_SYNC;
			break;
		default:
			usage();
		}
	}

	if (kilobytes) {
		chunk *= 1024;
	}

	if (nmegs == 0)
		usage();
	if (optind == argc)
		usage();
	filename = argv[optind++];
	if (optind != argc)
		usage();

	meg = round_up(_meg, 4096, align_offset);
	if (align_offset)
		printf("IO buffer is at %p\n", meg);

	fd = open(filename,
		o_sync|o_direct|O_RDWR|overwrite, 0666);
	if (fd < 0) {
		fprintf(stderr, "%s: cannot create %s: %s\n",
			argv[0], filename, strerror(errno));
		exit(1);
	}
	set_blocksize(fd);

	while (nmegs--) {
		int remainder = 1024 * 1024;

		while (remainder) {
			int to_write = chunk;

			if (to_write > remainder)
				to_write = remainder;

			if (write(fd, meg, to_write) != to_write) {
				perror("write");
				exit(1);
			}
			remainder -= to_write;
		}
	}

	if (do_fsync && fsync(fd) < 0) {
		fprintf(stderr, "%s: error in fsync: %s\n",
			argv[0], strerror(errno));
		exit(1);
	}
	close(fd);
	exit(0);
}
