/*
 * Filesystem test tool to stress the behaviour of long indirect block
 * chains.
 *
 * Creates sparse files and runs parallel writes() into holes to
 * populate the indirect chains.  Collisions between processes
 * allocating the same chains should exercise the kernel's retry paths
 * for indirect allocations. 
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/file.h>
#include <sys/wait.h>

int children = 20;
char *path   = "bigshared.tst";
int fd;

#define blocksize 4096
#define ptr_per_block (blocksize / 4)

void die_if_err(int err, const char *fmt, ...)
{
	static char error_buf[1024];
	va_list args;
	
	if (!err)
		return;

	va_start (args, fmt);
	vsnprintf (error_buf, 1024, fmt, args);
	va_end (args);

	fprintf(stderr, error_buf);
	
	exit (1);
}

typedef void (work_fn) (int);

void decode_exit(int, int);
void fork_child(int, work_fn *);
void sparse_writes(int);

int main(int argc, char *argv[])
{
	int i;
	
	fd = open(path, O_CREAT|O_RDWR|O_TRUNC, 0666);
	die_if_err(fd < 0, "open(%s)", path);

	for (i = 0; i < children; i++)
		fork_child(i, sparse_writes);
		
	for (;;) {
		pid_t pid;
		int status;
		
		/* Catch each child error status and report. */
		pid = wait3(&status, 0, 0);
		if (pid < 0)	/* No more children? */
			break;
		decode_exit(pid, status);
	}
	pause();
	return 0;
}


void decode_exit(int pid, int status)
{
	if (WIFEXITED (status)) {
		if (WEXITSTATUS (status))
			fprintf (stderr,
				 "Child %d exited with status %d\n",
				 pid, WEXITSTATUS(status));
		else {
			fprintf (stderr,
				 "Child %d exited normally\n",
				 pid);
		}
	} else {
		fprintf (stderr,
			 "Child %d exited with signal %d\n",
			 pid, WTERMSIG(status));
	}
}

void fork_child(int child, work_fn *fn)
{
	int pid = fork();
	die_if_err(pid < 0, "fork");
	if (pid)
		return;
	fn(child);
}

void do_write(int dind, int ind, int direct)
{
	static char buf[blocksize];
	int err;
	unsigned long offset;

	offset = ((ptr_per_block * ptr_per_block * dind) +
		  (ptr_per_block * ind) + direct + 12) * blocksize;

	err = pwrite(fd, buf, blocksize, offset);
	die_if_err(err != blocksize, "pwrite");
}

void sparse_writes(int child)
{
	/* Perform ordered writes into a sparse file.
	 *
	 * We will try to exercise double-indirect blocks one at a time,
	 * so that between passes over any given indirect block there is
	 * time for that block to be flushed from cache. 
	 *
	 * This assumes an on-disk blocksize of 4096 bytes. */

	int direct, ind, dind;

	setvbuf(stdout, NULL, _IONBF, 0);
	
	while (1) {
		if (child == 0) {
			printf("|");
			fsync(fd); 
		} else
			printf("+");
		for (ind = 0; ind < 10; ind++) {
			printf(".");
			/* This way there will be many allocating writes
			 * between passes over the same dind block. */
			for (dind = 0; dind < 200; dind++) {
				for (direct = 0; direct < 10; direct++)
					do_write(dind, ind, direct);
			}
		}
	}
}
